"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FunctionStreamingType = exports.FunctionType = exports.OrType = exports.RefType = exports.ObjectType = exports.ObjectOptionalFieldType = exports.ObjectFieldType = exports.TupleType = exports.ArrayType = exports.BinaryType = exports.StringType = exports.NumberType = exports.BooleanType = exports.ConstType = exports.AnyType = exports.AbstractType = void 0;
const tslib_1 = require("tslib");
const schema = tslib_1.__importStar(require("../schema"));
const util_1 = require("../util");
const json_clone_1 = require("../../json-clone");
const json_random_1 = require("../../json-random");
const json_binary_1 = require("../../json-binary");
const stringify_1 = require("../../json-text/stringify");
const printTree_1 = require("../../util/print/printTree");
const asString_1 = require("../../util/strings/asString");
const validate_1 = require("../schema/validate");
const ValidatorCodegenContext_1 = require("../codegen/validator/ValidatorCodegenContext");
const constants_1 = require("../constants");
const __deepEqual_1 = require("../../json-equal/$$deepEqual");
const normalizeAccessor_1 = require("../../util/codegen/util/normalizeAccessor");
const util_2 = require("../codegen/validator/util");
const JsonTextEncoderCodegenContext_1 = require("../codegen/json/JsonTextEncoderCodegenContext");
const CborEncoderCodegenContext_1 = require("../codegen/binary/CborEncoderCodegenContext");
const JsonEncoderCodegenContext_1 = require("../codegen/binary/JsonEncoderCodegenContext");
const CborEncoder_1 = require("../../json-pack/cbor/CborEncoder");
const JsExpression_1 = require("../../util/codegen/util/JsExpression");
const MessagePackEncoderCodegenContext_1 = require("../codegen/binary/MessagePackEncoderCodegenContext");
const msgpack_1 = require("../../json-pack/msgpack");
const lazyFunction_1 = require("../../util/lazyFunction");
const JsonEncoder_1 = require("../../json-pack/json/JsonEncoder");
const Writer_1 = require("../../util/buffers/Writer");
const CapacityEstimatorCodegenContext_1 = require("../codegen/capacity/CapacityEstimatorCodegenContext");
const json_size_1 = require("../../json-size");
const json_expression_1 = require("../../json-expression");
const operators_1 = require("../../json-expression/operators");
const Vars_1 = require("../../json-expression/Vars");
const augmentWithComment = (type, node) => {
    if (type.title || type.description) {
        let comment = '';
        if (type.title)
            comment += '# ' + type.title;
        if (type.title && type.description)
            comment += '\n\n';
        if (type.description)
            comment += type.description;
        node.comment = comment;
    }
};
class AbstractType {
    constructor() {
        this.validators = {};
        this.encoders = new Map();
    }
    getTypeName() {
        return this.schema.__t;
    }
    getSchema() {
        return this.schema;
    }
    getValidatorNames() {
        const { validator } = this.schema;
        if (!validator)
            return [];
        return Array.isArray(validator) ? validator : [validator];
    }
    toJsonSchema(ctx) {
        const schema = this.getSchema();
        const jsonSchema = {};
        if (schema.title)
            jsonSchema.title = schema.title;
        if (schema.description)
            jsonSchema.description = schema.description;
        if (schema.examples)
            jsonSchema.examples = schema.examples.map((example) => example.value);
        return jsonSchema;
    }
    options(options) {
        Object.assign(this.schema, options);
        return this;
    }
    getOptions() {
        const _a = this.schema, { __t } = _a, options = tslib_1.__rest(_a, ["__t"]);
        return options;
    }
    validate(value) {
        const validator = this.validator('string');
        const err = validator(value);
        if (err)
            throw new Error(JSON.parse(err)[0]);
    }
    compileValidator(options) {
        const ctx = new ValidatorCodegenContext_1.ValidatorCodegenContext(Object.assign(Object.assign({ system: this.system, errors: 'object' }, options), { type: this }));
        this.codegenValidator(ctx, [], ctx.codegen.options.args[0]);
        return ctx.compile();
    }
    __compileValidator(kind) {
        return (this.validators[kind] = this.compileValidator({
            errors: kind,
            system: this.system,
            skipObjectExtraFieldsCheck: kind === 'boolean',
            unsafeMode: kind === 'boolean',
        }));
    }
    validator(kind) {
        return this.validators[kind] || (0, lazyFunction_1.lazy)(() => this.__compileValidator(kind));
    }
    compileJsonTextEncoder(options) {
        const ctx = new JsonTextEncoderCodegenContext_1.JsonTextEncoderCodegenContext(Object.assign(Object.assign({}, options), { system: this.system, type: this }));
        const r = ctx.codegen.options.args[0];
        const value = new JsExpression_1.JsExpression(() => r);
        this.codegenJsonTextEncoder(ctx, value);
        return ctx.compile();
    }
    codegenJsonTextEncoder(ctx, value) {
        throw new Error(`${this.constructor.name}.codegenJsonTextEncoder() not implemented`);
    }
    jsonTextEncoder() {
        return (this.__jsonEncoder || (this.__jsonEncoder = (0, lazyFunction_1.lazy)(() => (this.__jsonEncoder = this.compileJsonTextEncoder({})))));
    }
    compileEncoder(format, name) {
        switch (format) {
            case 0: {
                const encoder = this.compileCborEncoder({ name });
                this.encoders.set(0, encoder);
                return encoder;
            }
            case 1: {
                const encoder = this.compileMessagePackEncoder({ name });
                this.encoders.set(1, encoder);
                return encoder;
            }
            case 2: {
                const encoder = this.compileJsonEncoder({ name });
                this.encoders.set(2, encoder);
                return encoder;
            }
            default:
                throw new Error(`Unsupported encoding format: ${format}`);
        }
    }
    encoder(kind) {
        const encoders = this.encoders;
        const cachedEncoder = encoders.get(kind);
        if (cachedEncoder)
            return cachedEncoder;
        const temporaryWrappedEncoder = (0, lazyFunction_1.lazy)(() => this.compileEncoder(kind));
        encoders.set(kind, temporaryWrappedEncoder);
        return temporaryWrappedEncoder;
    }
    encode(codec, value) {
        const encoder = this.encoder(codec.format);
        const writer = codec.encoder.writer;
        writer.reset();
        encoder(value, codec.encoder);
        return writer.flush();
    }
    codegenValidator(ctx, path, r) {
        throw new Error(`${this.constructor.name}.codegenValidator() not implemented`);
    }
    compileCborEncoder(options) {
        const ctx = new CborEncoderCodegenContext_1.CborEncoderCodegenContext(Object.assign(Object.assign({ system: this.system, encoder: new CborEncoder_1.CborEncoder() }, options), { type: this }));
        const r = ctx.codegen.options.args[0];
        const value = new JsExpression_1.JsExpression(() => r);
        this.codegenCborEncoder(ctx, value);
        return ctx.compile();
    }
    codegenCborEncoder(ctx, value) {
        throw new Error(`${this.constructor.name}.codegenCborEncoder() not implemented`);
    }
    compileMessagePackEncoder(options) {
        const ctx = new MessagePackEncoderCodegenContext_1.MessagePackEncoderCodegenContext(Object.assign(Object.assign({ system: this.system, encoder: new msgpack_1.MsgPackEncoder() }, options), { type: this }));
        const r = ctx.codegen.options.args[0];
        const value = new JsExpression_1.JsExpression(() => r);
        this.codegenMessagePackEncoder(ctx, value);
        return ctx.compile();
    }
    codegenMessagePackEncoder(ctx, value) {
        throw new Error(`${this.constructor.name}.codegenMessagePackEncoder() not implemented`);
    }
    compileJsonEncoder(options) {
        const writer = new Writer_1.Writer();
        const ctx = new JsonEncoderCodegenContext_1.JsonEncoderCodegenContext(Object.assign(Object.assign({ system: this.system, encoder: new JsonEncoder_1.JsonEncoder(writer) }, options), { type: this }));
        const r = ctx.codegen.options.args[0];
        const value = new JsExpression_1.JsExpression(() => r);
        this.codegenJsonEncoder(ctx, value);
        return ctx.compile();
    }
    codegenJsonEncoder(ctx, value) {
        throw new Error(`${this.constructor.name}.codegenJsonEncoder() not implemented`);
    }
    compileCapacityEstimator(options) {
        const ctx = new CapacityEstimatorCodegenContext_1.CapacityEstimatorCodegenContext(Object.assign(Object.assign({ system: this.system }, options), { type: this }));
        const r = ctx.codegen.options.args[0];
        const value = new JsExpression_1.JsExpression(() => r);
        this.codegenCapacityEstimator(ctx, value);
        return ctx.compile();
    }
    codegenCapacityEstimator(ctx, value) {
        throw new Error(`${this.constructor.name}.codegenCapacityEstimator() not implemented`);
    }
    capacityEstimator() {
        return (this.__capacityEstimator ||
            (this.__capacityEstimator = (0, lazyFunction_1.lazy)(() => (this.__capacityEstimator = this.compileCapacityEstimator({})))));
    }
    random() {
        return json_random_1.RandomJson.generate({ nodeCount: 5 });
    }
    toTypeScriptAst() {
        const node = { node: 'UnknownKeyword' };
        return node;
    }
    toJson(value, system = this.system) {
        return JSON.stringify(value);
    }
    toStringTitle() {
        return this.getTypeName();
    }
    toStringOptions() {
        const options = this.getOptions();
        if (Object.keys(options).length === 0)
            return '';
        return (0, stringify_1.stringify)(options);
    }
    toString(tab = '') {
        const options = this.toStringOptions();
        return this.toStringTitle() + (options ? ` ${options}` : '');
    }
}
exports.AbstractType = AbstractType;
class AnyType extends AbstractType {
    constructor(schema) {
        super();
        this.schema = schema;
    }
    toJsonSchema(ctx) {
        return Object.assign({ type: ['string', 'number', 'boolean', 'null', 'array', 'object'] }, super.toJsonSchema(ctx));
    }
    validateSchema() {
        (0, validate_1.validateTType)(this.getSchema(), 'any');
    }
    codegenValidator(ctx, path, r) {
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.js(`s += stringify(${value.use()});`);
    }
    codegenBinaryEncoder(ctx, value) {
        ctx.codegen.link('Value');
        const r = ctx.codegen.var(value.use());
        ctx.codegen.if(`${r} instanceof Value`, () => {
            ctx.codegen.if(`${r}.type`, () => {
                const type = ctx instanceof CborEncoderCodegenContext_1.CborEncoderCodegenContext
                    ? 0
                    : ctx instanceof MessagePackEncoderCodegenContext_1.MessagePackEncoderCodegenContext
                        ? 1
                        : 2;
                ctx.js(`${r}.type.encoder(${type})(${r}.data, encoder);`);
            }, () => {
                ctx.js(`encoder.writeAny(${r}.data);`);
            });
        }, () => {
            ctx.js(`encoder.writeAny(${r});`);
        });
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        const codegen = ctx.codegen;
        codegen.link('Value');
        const r = codegen.var(value.use());
        codegen.if(`${r} instanceof Value`, () => {
            codegen.if(`${r}.type`, () => {
                ctx.codegen.js(`size += ${r}.type.capacityEstimator()(${r}.data);`);
            }, () => {
                ctx.codegen.js(`size += maxEncodingCapacity(${r}.data);`);
            });
        }, () => {
            ctx.codegen.js(`size += maxEncodingCapacity(${r});`);
        });
    }
    random() {
        return json_random_1.RandomJson.generate({ nodeCount: 5 });
    }
    toTypeScriptAst() {
        return { node: 'AnyKeyword' };
    }
}
exports.AnyType = AnyType;
class ConstType extends AbstractType {
    constructor(schema) {
        super();
        this.schema = schema;
        this.__json = JSON.stringify(schema.value);
    }
    value() {
        return this.schema.value;
    }
    toJsonSchema(ctx) {
        const schema = this.schema;
        return Object.assign({ type: typeof this.schema.value, const: schema.value }, super.toJsonSchema(ctx));
    }
    getOptions() {
        const _a = this.schema, { __t, value } = _a, options = tslib_1.__rest(_a, ["__t", "value"]);
        return options;
    }
    validateSchema() {
        (0, validate_1.validateTType)(this.getSchema(), 'const');
    }
    codegenValidator(ctx, path, r) {
        const value = this.schema.value;
        const equals = (0, __deepEqual_1.$$deepEqual)(value);
        const fn = ctx.codegen.addConstant(equals);
        ctx.js(`if (!${fn}(${r})) return ${ctx.err(constants_1.ValidationError.CONST, path)}`);
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.writeText(JSON.stringify(this.schema.value));
    }
    codegenBinaryEncoder(ctx, value) {
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeAny(this.schema.value);
        }));
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        ctx.inc((0, json_size_1.maxEncodingCapacity)(this.value()));
    }
    random() {
        return (0, json_clone_1.cloneBinary)(this.schema.value);
    }
    toTypeScriptAst() {
        const value = this.schema.value;
        if (value === null) {
            const node = { node: 'NullKeyword' };
            return node;
        }
        switch (typeof value) {
            case 'string': {
                const node = { node: 'StringLiteral', text: value };
                return node;
            }
            case 'number': {
                const node = { node: 'NumericLiteral', text: value.toString() };
                return node;
            }
            case 'boolean': {
                const node = { node: value ? 'TrueKeyword' : 'FalseKeyword' };
                return node;
            }
            case 'object': {
                const node = { node: 'ObjectKeyword' };
                return node;
            }
            default: {
                const node = { node: 'UnknownKeyword' };
                return node;
            }
        }
    }
    toJson(value, system = this.system) {
        return this.__json;
    }
    toString(tab = '') {
        return `${super.toString(tab)} → ${JSON.stringify(this.schema.value)}`;
    }
}
exports.ConstType = ConstType;
class BooleanType extends AbstractType {
    constructor(schema) {
        super();
        this.schema = schema;
    }
    toJsonSchema(ctx) {
        return Object.assign({ type: 'boolean' }, super.toJsonSchema(ctx));
    }
    validateSchema() {
        (0, validate_1.validateTType)(this.getSchema(), 'bool');
    }
    codegenValidator(ctx, path, r) {
        const err = ctx.err(constants_1.ValidationError.BOOL, path);
        ctx.js(`if(typeof ${r} !== "boolean") return ${err};`);
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.js(`s += ${value.use()} ? 'true' : 'false';`);
    }
    codegenBinaryEncoder(ctx, value) {
        ctx.js(`encoder.writeBoolean(${value.use()});`);
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        ctx.inc(5);
    }
    random() {
        return json_random_1.RandomJson.genBoolean();
    }
    toTypeScriptAst() {
        return { node: 'BooleanKeyword' };
    }
    toJson(value, system = this.system) {
        return (value ? 'true' : 'false');
    }
}
exports.BooleanType = BooleanType;
class NumberType extends AbstractType {
    constructor(schema) {
        super();
        this.schema = schema;
    }
    toJsonSchema(ctx) {
        const schema = this.getSchema();
        const jsonSchema = Object.assign({ type: 'number' }, super.toJsonSchema(ctx));
        if (schema.format && util_1.ints.has(schema.format))
            jsonSchema.type = 'integer';
        if (schema.gt !== undefined)
            jsonSchema.exclusiveMinimum = schema.gt;
        if (schema.gte !== undefined)
            jsonSchema.minimum = schema.gte;
        if (schema.lt !== undefined)
            jsonSchema.exclusiveMaximum = schema.lt;
        if (schema.lte !== undefined)
            jsonSchema.maximum = schema.lte;
        return jsonSchema;
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'num');
        (0, validate_1.validateWithValidator)(schema);
        const { format, gt, gte, lt, lte } = schema;
        if (gt !== undefined && typeof gt !== 'number')
            throw new Error('GT_TYPE');
        if (gte !== undefined && typeof gte !== 'number')
            throw new Error('GTE_TYPE');
        if (lt !== undefined && typeof lt !== 'number')
            throw new Error('LT_TYPE');
        if (lte !== undefined && typeof lte !== 'number')
            throw new Error('LTE_TYPE');
        if (gt !== undefined && gte !== undefined)
            throw new Error('GT_GTE');
        if (lt !== undefined && lte !== undefined)
            throw new Error('LT_LTE');
        if ((gt !== undefined || gte !== undefined) && (lt !== undefined || lte !== undefined))
            if ((gt !== null && gt !== void 0 ? gt : gte) > (lt !== null && lt !== void 0 ? lt : lte))
                throw new Error('GT_LT');
        if (format !== undefined) {
            if (typeof format !== 'string')
                throw new Error('FORMAT_TYPE');
            if (!format)
                throw new Error('FORMAT_EMPTY');
            switch (format) {
                case 'i':
                case 'u':
                case 'f':
                case 'i8':
                case 'i16':
                case 'i32':
                case 'i64':
                case 'u8':
                case 'u16':
                case 'u32':
                case 'u64':
                case 'f32':
                case 'f64':
                    break;
                default:
                    throw new Error('FORMAT_INVALID');
            }
        }
    }
    codegenValidator(ctx, path, r) {
        const { format, gt, gte, lt, lte } = this.schema;
        if (format && util_1.ints.has(format)) {
            const errInt = ctx.err(constants_1.ValidationError.INT, path);
            ctx.js(`if(!Number.isInteger(${r})) return ${errInt};`);
            if (util_1.uints.has(format)) {
                const err = ctx.err(constants_1.ValidationError.UINT, path);
                ctx.js(`if(${r} < 0) return ${err};`);
                switch (format) {
                    case 'u8': {
                        ctx.js(`if(${r} > 0xFF) return ${err};`);
                        break;
                    }
                    case 'u16': {
                        ctx.js(`if(${r} > 0xFFFF) return ${err};`);
                        break;
                    }
                    case 'u32': {
                        ctx.js(`if(${r} > 0xFFFFFFFF) return ${err};`);
                        break;
                    }
                }
            }
            else {
                switch (format) {
                    case 'i8': {
                        ctx.js(`if(${r} > 0x7F || ${r} < -0x80) return ${errInt};`);
                        break;
                    }
                    case 'i16': {
                        ctx.js(`if(${r} > 0x7FFF || ${r} < -0x8000) return ${errInt};`);
                        break;
                    }
                    case 'i32': {
                        ctx.js(`if(${r} > 0x7FFFFFFF || ${r} < -0x80000000) return ${errInt};`);
                        break;
                    }
                }
            }
        }
        else if (util_1.floats.has(format)) {
            const err = ctx.err(constants_1.ValidationError.NUM, path);
            ctx.codegen.js(`if(!Number.isFinite(${r})) return ${err};`);
        }
        else {
            const err = ctx.err(constants_1.ValidationError.NUM, path);
            ctx.codegen.js(`if(typeof ${r} !== "number") return ${err};`);
        }
        if (gt !== undefined) {
            const err = ctx.err(constants_1.ValidationError.GT, path);
            ctx.codegen.js(`if(${r} <= ${gt}) return ${err};`);
        }
        if (gte !== undefined) {
            const err = ctx.err(constants_1.ValidationError.GTE, path);
            ctx.codegen.js(`if(${r} < ${gte}) return ${err};`);
        }
        if (lt !== undefined) {
            const err = ctx.err(constants_1.ValidationError.LT, path);
            ctx.codegen.js(`if(${r} >= ${lt}) return ${err};`);
        }
        if (lte !== undefined) {
            const err = ctx.err(constants_1.ValidationError.LTE, path);
            ctx.codegen.js(`if(${r} > ${lte}) return ${err};`);
        }
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.js(`s += ${value.use()};`);
    }
    codegenBinaryEncoder(ctx, value) {
        const { format } = this.schema;
        const v = value.use();
        if (util_1.uints.has(format))
            ctx.js(`encoder.writeUInteger(${v});`);
        else if (util_1.ints.has(format))
            ctx.js(`encoder.writeInteger(${v});`);
        else if (util_1.floats.has(format))
            ctx.js(`encoder.writeFloat(${v});`);
        else
            ctx.js(`encoder.writeNumber(${v});`);
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        ctx.inc(22);
    }
    random() {
        let num = Math.random();
        let min = Number.MIN_SAFE_INTEGER;
        let max = Number.MAX_SAFE_INTEGER;
        if (this.schema.gt !== undefined)
            min = this.schema.gt;
        if (this.schema.gte !== undefined)
            min = this.schema.gte + 0.000000000000001;
        if (this.schema.lt !== undefined)
            max = this.schema.lt;
        if (this.schema.lte !== undefined)
            max = this.schema.lte - 0.000000000000001;
        if (this.schema.format) {
            switch (this.schema.format) {
                case 'i8':
                    min = Math.max(min, -0x80);
                    max = Math.min(max, 0x7f);
                    break;
                case 'i16':
                    min = Math.max(min, -0x8000);
                    max = Math.min(max, 0x7fff);
                    break;
                case 'i32':
                    min = Math.max(min, -0x80000000);
                    max = Math.min(max, 0x7fffffff);
                    break;
                case 'i64':
                case 'i':
                    min = Math.max(min, -0x8000000000);
                    max = Math.min(max, 0x7fffffffff);
                    break;
                case 'u8':
                    min = Math.max(min, 0);
                    max = Math.min(max, 0xff);
                    break;
                case 'u16':
                    min = Math.max(min, 0);
                    max = Math.min(max, 0xffff);
                    break;
                case 'u32':
                    min = Math.max(min, 0);
                    max = Math.min(max, 0xffffffff);
                    break;
                case 'u64':
                case 'u':
                    min = Math.max(min, 0);
                    max = Math.min(max, 0xffffffffffff);
                    break;
            }
            return Math.round(num * (max - min)) + min;
        }
        num = num * (max - min) + min;
        if (Math.random() > 0.7)
            num = Math.round(num);
        if (num === -0)
            return 0;
        return num;
    }
    toTypeScriptAst() {
        return { node: 'NumberKeyword' };
    }
    toJson(value, system = this.system) {
        return ('' + value);
    }
}
exports.NumberType = NumberType;
class StringType extends AbstractType {
    constructor(schema) {
        super();
        this.schema = schema;
    }
    toJsonSchema(ctx) {
        const schema = this.getSchema();
        const jsonSchema = Object.assign({ type: 'string' }, super.toJsonSchema(ctx));
        if (schema.min !== undefined)
            jsonSchema.minLength = schema.min;
        if (schema.max !== undefined)
            jsonSchema.maxLength = schema.max;
        return jsonSchema;
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'str');
        (0, validate_1.validateWithValidator)(schema);
        const { min, max, ascii, noJsonEscape } = schema;
        (0, validate_1.validateMinMax)(min, max);
        if (ascii !== undefined) {
            if (typeof ascii !== 'boolean')
                throw new Error('ASCII');
        }
        if (noJsonEscape !== undefined) {
            if (typeof noJsonEscape !== 'boolean')
                throw new Error('NO_JSON_ESCAPE_TYPE');
        }
    }
    codegenValidator(ctx, path, r) {
        const error = ctx.err(constants_1.ValidationError.STR, path);
        ctx.js(`if(typeof ${r} !== "string") return ${error};`);
        const { min, max } = this.schema;
        if (typeof min === 'number' && min === max) {
            const err = ctx.err(constants_1.ValidationError.STR_LEN, path);
            ctx.js(`if(${r}.length !== ${min}) return ${err};`);
        }
        else {
            if (typeof min === 'number') {
                const err = ctx.err(constants_1.ValidationError.STR_LEN, path);
                ctx.js(`if(${r}.length < ${min}) return ${err};`);
            }
            if (typeof max === 'number') {
                const err = ctx.err(constants_1.ValidationError.STR_LEN, path);
                ctx.js(`if(${r}.length > ${max}) return ${err};`);
            }
        }
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        if (this.schema.noJsonEscape) {
            ctx.writeText('"');
            ctx.js(`s += ${value.use()};`);
            ctx.writeText('"');
        }
        else
            ctx.js(`s += asString(${value.use()});`);
    }
    codegenBinaryEncoder(ctx, value) {
        const ascii = this.schema.ascii;
        const v = value.use();
        if (ascii)
            ctx.js(`encoder.writeAsciiStr(${v});`);
        else
            ctx.js(`encoder.writeStr(${v});`);
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        ctx.inc(5);
        ctx.codegen.js(`size += ${5} * ${value.use()}.length;`);
    }
    random() {
        let length = Math.round(Math.random() * 10);
        const { min, max } = this.schema;
        if (min !== undefined && length < min)
            length = min + length;
        if (max !== undefined && length > max)
            length = max;
        return json_random_1.RandomJson.genString(length);
    }
    toTypeScriptAst() {
        return { node: 'StringKeyword' };
    }
    toJson(value, system = this.system) {
        return (this.schema.noJsonEscape ? '"' + value + '"' : (0, asString_1.asString)(value));
    }
}
exports.StringType = StringType;
class BinaryType extends AbstractType {
    constructor(type, options) {
        super();
        this.type = type;
        this.schema = schema.s.Binary(schema.s.any, options);
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { type: this.type.getSchema() });
    }
    toJsonSchema(ctx) {
        return Object.assign({ type: 'binary' }, super.toJsonSchema(ctx));
    }
    getOptions() {
        const _a = this.schema, { __t, type } = _a, options = tslib_1.__rest(_a, ["__t", "type"]);
        return options;
    }
    validateSchema() {
        (0, validate_1.validateTType)(this.getSchema(), 'bin');
        this.type.validateSchema();
    }
    codegenValidator(ctx, path, r) {
        const hasBuffer = typeof Buffer === 'function';
        const err = ctx.err(constants_1.ValidationError.BIN, path);
        ctx.js(`if(!(${r} instanceof Uint8Array)${hasBuffer ? ` && !Buffer.isBuffer(${r})` : ''}) return ${err};`);
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.linkBase64();
        ctx.writeText('"data:application/octet-stream;base64,');
        ctx.js(`s += toBase64(${value.use()});`);
        ctx.writeText('"');
    }
    codegenBinaryEncoder(ctx, value) {
        ctx.js(`encoder.writeBin(${value.use()});`);
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        ctx.inc(41);
        ctx.codegen.js(`size += ${2} * ${value.use()}.length;`);
    }
    random() {
        const octets = json_random_1.RandomJson.genString()
            .split('')
            .map((c) => c.charCodeAt(0));
        return new Uint8Array(octets);
    }
    toTypeScriptAst() {
        return {
            node: 'GenericTypeAnnotation',
            id: {
                node: 'Identifier',
                name: 'Uint8Array',
            },
        };
    }
    toJson(value, system = this.system) {
        return ('"' + (0, json_binary_1.stringifyBinary)(value) + '"');
    }
    toString(tab = '') {
        return super.toString(tab) + (0, printTree_1.printTree)(tab, [(tab) => this.type.toString(tab)]);
    }
}
exports.BinaryType = BinaryType;
class ArrayType extends AbstractType {
    constructor(type, options) {
        super();
        this.type = type;
        this.schema = schema.s.Array(schema.s.any, options);
    }
    getSchema(ctx) {
        return Object.assign(Object.assign({}, this.schema), { type: this.type.getSchema(ctx) });
    }
    toJsonSchema() {
        const schema = this.getSchema();
        const jsonSchema = Object.assign({ type: 'array', items: this.type.toJsonSchema() }, super.toJsonSchema());
        if (schema.min !== undefined)
            jsonSchema.minItems = schema.min;
        if (schema.max !== undefined)
            jsonSchema.maxItems = schema.max;
        return jsonSchema;
    }
    getOptions() {
        const _a = this.schema, { __t, type } = _a, options = tslib_1.__rest(_a, ["__t", "type"]);
        return options;
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'arr');
        const { min, max } = schema;
        (0, validate_1.validateMinMax)(min, max);
        this.type.validateSchema();
    }
    codegenValidator(ctx, path, r) {
        const rl = ctx.codegen.getRegister();
        const ri = ctx.codegen.getRegister();
        const rv = ctx.codegen.getRegister();
        const err = ctx.err(constants_1.ValidationError.ARR, path);
        const errLen = ctx.err(constants_1.ValidationError.ARR_LEN, path);
        const { min, max } = this.schema;
        ctx.js(`if (!Array.isArray(${r})) return ${err};`);
        ctx.js(`var ${rl} = ${r}.length;`);
        if (min !== undefined)
            ctx.js(`if (${rl} < ${min}) return ${errLen};`);
        if (max !== undefined)
            ctx.js(`if (${rl} > ${max}) return ${errLen};`);
        ctx.js(`for (var ${rv}, ${ri} = ${r}.length; ${ri}-- !== 0;) {`);
        ctx.js(`${rv} = ${r}[${ri}];`);
        this.type.codegenValidator(ctx, [...path, { r: ri }], rv);
        ctx.js(`}`);
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.writeText('[');
        const codegen = ctx.codegen;
        const r = codegen.getRegister();
        const rl = codegen.getRegister();
        const rll = codegen.getRegister();
        const ri = codegen.getRegister();
        ctx.js(`var ${r} = ${value.use()}, ${rl} = ${r}.length, ${rll} = ${rl} - 1, ${ri} = 0;`);
        ctx.js(`for(; ${ri} < ${rll}; ${ri}++) ` + '{');
        this.type.codegenJsonTextEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}[${ri}]`));
        ctx.js(`s += ',';`);
        ctx.js(`}`);
        ctx.js(`if (${rl}) {`);
        this.type.codegenJsonTextEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}[${rll}]`));
        ctx.js(`}`);
        ctx.writeText(']');
    }
    codegenBinaryEncoder(ctx, value) {
        const type = this.type;
        const codegen = ctx.codegen;
        const r = codegen.getRegister();
        const rl = codegen.getRegister();
        const ri = codegen.getRegister();
        const rItem = codegen.getRegister();
        const expr = new JsExpression_1.JsExpression(() => `${rItem}`);
        ctx.js(`var ${r} = ${value.use()}, ${rl} = ${r}.length, ${ri} = 0, ${rItem};`);
        ctx.js(`encoder.writeArrHdr(${rl});`);
        ctx.js(`for(; ${ri} < ${rl}; ${ri}++) ` + '{');
        ctx.js(`${rItem} = ${r}[${ri}];`);
        if (ctx instanceof CborEncoderCodegenContext_1.CborEncoderCodegenContext)
            type.codegenCborEncoder(ctx, expr);
        else if (ctx instanceof MessagePackEncoderCodegenContext_1.MessagePackEncoderCodegenContext)
            type.codegenMessagePackEncoder(ctx, expr);
        else
            throw new Error('Unknown encoder');
        ctx.js(`}`);
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        const type = this.type;
        const codegen = ctx.codegen;
        const expr = new JsExpression_1.JsExpression(() => `${rItem}`);
        const r = codegen.var(value.use());
        const rLen = codegen.var(`${r}.length`);
        const rLast = codegen.var(`${rLen} - 1`);
        const ri = codegen.var('0');
        const rItem = codegen.var();
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeStartArr();
        }));
        codegen.js(`for(; ${ri} < ${rLast}; ${ri}++) {`);
        codegen.js(`${rItem} = ${r}[${ri}];`);
        type.codegenJsonEncoder(ctx, expr);
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeArrSeparator();
        }));
        ctx.js(`}`);
        ctx.js(`if (${rLen}) {`);
        codegen.js(`${rItem} = ${r}[${rLast}];`);
        type.codegenJsonEncoder(ctx, expr);
        ctx.js(`}`);
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeEndArr();
        }));
    }
    codegenCapacityEstimator(ctx, value) {
        const codegen = ctx.codegen;
        ctx.inc(5);
        const rLen = codegen.var(`${value.use()}.length`);
        const type = this.type;
        codegen.js(`size += ${1 === 1 ? `${rLen}` : `${1} * ${rLen}`};`);
        const fn = type.compileCapacityEstimator({
            system: ctx.options.system,
            name: ctx.options.name,
        });
        const isConstantSizeType = type instanceof ConstType || type instanceof BooleanType || type instanceof NumberType;
        if (isConstantSizeType) {
            codegen.js(`size += ${rLen} * ${fn(null)};`);
        }
        else {
            const r = codegen.var(value.use());
            const rFn = codegen.linkDependency(fn);
            const ri = codegen.getRegister();
            codegen.js(`for(var ${ri} = ${rLen}; ${ri}-- !== 0;) size += ${rFn}(${r}[${ri}]);`);
        }
    }
    random() {
        let length = Math.round(Math.random() * 10);
        const { min, max } = this.schema;
        if (min !== undefined && length < min)
            length = min + length;
        if (max !== undefined && length > max)
            length = max;
        const arr = [];
        for (let i = 0; i < length; i++)
            arr.push(this.type.random());
        return arr;
    }
    toTypeScriptAst() {
        return {
            node: 'ArrayType',
            elementType: this.type.toTypeScriptAst(),
        };
    }
    toJson(value, system = this.system) {
        const length = value.length;
        if (!length)
            return '[]';
        const last = length - 1;
        const type = this.type;
        let str = '[';
        for (let i = 0; i < last; i++)
            str += type.toJson(value[i], system) + ',';
        str += type.toJson(value[last], system);
        return (str + ']');
    }
    toString(tab = '') {
        return super.toString(tab) + (0, printTree_1.printTree)(tab, [(tab) => this.type.toString(tab)]);
    }
}
exports.ArrayType = ArrayType;
class TupleType extends AbstractType {
    constructor(types, options) {
        super();
        this.types = types;
        this.schema = Object.assign(Object.assign({}, schema.s.Tuple()), options);
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { types: this.types.map((type) => type.getSchema()) });
    }
    toJsonSchema(ctx) {
        const jsonSchema = Object.assign({ type: 'array', prefixItems: this.types.map((type) => type.toJsonSchema(ctx)), items: false }, super.toJsonSchema(ctx));
        return jsonSchema;
    }
    getOptions() {
        const _a = this.schema, { __t, types } = _a, options = tslib_1.__rest(_a, ["__t", "types"]);
        return options;
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'tup');
        const { types } = schema;
        if (!Array.isArray(types))
            throw new Error('TYPES_TYPE');
        if (!types.length)
            throw new Error('TYPES_LENGTH');
        for (const type of this.types)
            type.validateSchema();
    }
    codegenValidator(ctx, path, r) {
        const err = ctx.err(constants_1.ValidationError.TUP, path);
        const types = this.types;
        ctx.js(`if (!Array.isArray(${r}) || ${r}.length !== ${types.length}) return ${err};`);
        for (let i = 0; i < this.types.length; i++) {
            const rv = ctx.codegen.getRegister();
            ctx.js(`var ${rv} = ${r}[${i}];`);
            types[i].codegenValidator(ctx, [...path, i], rv);
        }
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.writeText('[');
        const types = this.types;
        const length = types.length;
        const last = length - 1;
        for (let i = 0; i < last; i++) {
            types[i].codegenJsonTextEncoder(ctx, new JsExpression_1.JsExpression(() => `${value.use()}[${i}]`));
            ctx.writeText(',');
        }
        types[last].codegenJsonTextEncoder(ctx, new JsExpression_1.JsExpression(() => `${value.use()}[${last}]`));
        ctx.writeText(']');
    }
    codegenBinaryEncoder(ctx, value) {
        const types = this.types;
        const length = types.length;
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeArrHdr(length);
        }));
        const r = ctx.codegen.r();
        ctx.js(`var ${r} = ${value.use()};`);
        for (let i = 0; i < length; i++)
            if (ctx instanceof CborEncoderCodegenContext_1.CborEncoderCodegenContext)
                types[i].codegenCborEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}[${i}]`));
            else
                types[i].codegenMessagePackEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}[${i}]`));
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        const codegen = ctx.codegen;
        const expr = new JsExpression_1.JsExpression(() => `${rItem}`);
        const r = codegen.var(value.use());
        const rItem = codegen.var();
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeStartArr();
        }));
        const types = this.types;
        const length = types.length;
        const arrSepBlob = ctx.gen((encoder) => {
            encoder.writeArrSeparator();
        });
        for (let i = 0; i < length; i++) {
            const type = types[i];
            const isLast = i === length - 1;
            codegen.js(`${rItem} = ${r}[${i}];`);
            type.codegenJsonEncoder(ctx, expr);
            if (!isLast)
                ctx.blob(arrSepBlob);
        }
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeEndArr();
        }));
    }
    codegenCapacityEstimator(ctx, value) {
        const codegen = ctx.codegen;
        const r = codegen.var(value.use());
        const types = this.types;
        const overhead = 5 + 1 * types.length;
        ctx.inc(overhead);
        for (let i = 0; i < types.length; i++) {
            const type = types[i];
            const fn = type.compileCapacityEstimator({
                system: ctx.options.system,
                name: ctx.options.name,
            });
            const rFn = codegen.linkDependency(fn);
            codegen.js(`size += ${rFn}(${r}[${i}]);`);
        }
    }
    random() {
        return this.types.map((type) => type.random());
    }
    toTypeScriptAst() {
        return {
            node: 'TupleType',
            elements: this.types.map((type) => type.toTypeScriptAst()),
        };
    }
    toJson(value, system = this.system) {
        const types = this.types;
        const length = types.length;
        if (!length)
            return '[]';
        const last = length - 1;
        let str = '[';
        for (let i = 0; i < last; i++)
            str += types[i].toJson(value[i], system) + ',';
        str += types[last].toJson(value[last], system);
        return (str + ']');
    }
    toString(tab = '') {
        return super.toString(tab) + (0, printTree_1.printTree)(tab, [...this.types.map((type) => (tab) => type.toString(tab))]);
    }
}
exports.TupleType = TupleType;
class ObjectFieldType extends AbstractType {
    constructor(key, value) {
        super();
        this.key = key;
        this.value = value;
        this.schema = schema.s.prop(key, schema.s.any);
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { type: this.value.getSchema() });
    }
    getOptions() {
        const _a = this.schema, { __t, key, type, optional } = _a, options = tslib_1.__rest(_a, ["__t", "key", "type", "optional"]);
        return options;
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'field');
        const { key, optional } = schema;
        if (typeof key !== 'string')
            throw new Error('KEY_TYPE');
        if (optional !== undefined && typeof optional !== 'boolean')
            throw new Error('OPTIONAL_TYPE');
        this.value.validateSchema();
    }
    toStringTitle() {
        return `"${this.key}":`;
    }
    toString(tab = '') {
        return super.toString(tab) + (0, printTree_1.printTree)(tab + ' ', [(tab) => this.value.toString(tab)]);
    }
}
exports.ObjectFieldType = ObjectFieldType;
class ObjectOptionalFieldType extends ObjectFieldType {
    constructor(key, value) {
        super(key, value);
        this.key = key;
        this.value = value;
        this.optional = true;
        this.schema = schema.s.propOpt(key, schema.s.any);
    }
    toStringTitle() {
        return `"${this.key}"?:`;
    }
}
exports.ObjectOptionalFieldType = ObjectOptionalFieldType;
class ObjectType extends AbstractType {
    constructor(fields) {
        super();
        this.fields = fields;
        this.schema = schema.s.obj;
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { fields: this.fields.map((f) => f.getSchema()) });
    }
    toJsonSchema(ctx) {
        const jsonSchema = Object.assign({ type: 'object', properties: {} }, super.toJsonSchema(ctx));
        const required = [];
        for (const field of this.fields) {
            jsonSchema.properties[field.key] = field.value.toJsonSchema(ctx);
            if (!(field instanceof ObjectOptionalFieldType))
                required.push(field.key);
        }
        if (required.length)
            jsonSchema.required = required;
        if (this.schema.unknownFields === false)
            jsonSchema.additionalProperties = false;
        return jsonSchema;
    }
    getOptions() {
        const _a = this.schema, { __t, fields } = _a, options = tslib_1.__rest(_a, ["__t", "fields"]);
        return options;
    }
    getField(key) {
        return this.fields.find((f) => f.key === key);
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'obj');
        (0, validate_1.validateWithValidator)(schema);
        const { fields, unknownFields } = schema;
        if (!Array.isArray(fields))
            throw new Error('FIELDS_TYPE');
        if (unknownFields !== undefined && typeof unknownFields !== 'boolean')
            throw new Error('UNKNOWN_FIELDS_TYPE');
        for (const field of this.fields)
            field.validateSchema();
    }
    codegenValidator(ctx, path, r) {
        const fields = this.fields;
        const length = fields.length;
        const canSkipObjectTypeCheck = ctx.options.unsafeMode && length > 0;
        if (!canSkipObjectTypeCheck) {
            const err = ctx.err(constants_1.ValidationError.OBJ, path);
            ctx.js(`if (typeof ${r} !== 'object' || !${r} || (${r} instanceof Array)) return ${err};`);
        }
        const checkExtraKeys = length && !this.schema.unknownFields && !ctx.options.skipObjectExtraFieldsCheck;
        if (checkExtraKeys) {
            const rk = ctx.codegen.getRegister();
            ctx.js(`for (var ${rk} in ${r}) {`);
            ctx.js(`switch (${rk}) { case ${fields
                .map((field) => JSON.stringify(field.key))
                .join(': case ')}: break; default: return ${ctx.err(constants_1.ValidationError.KEYS, [...path, { r: rk }])};}`);
            ctx.js(`}`);
        }
        for (let i = 0; i < length; i++) {
            const field = fields[i];
            const rv = ctx.codegen.getRegister();
            const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
            const keyPath = [...path, field.key];
            if (field instanceof ObjectOptionalFieldType) {
                ctx.js(`var ${rv} = ${r}${accessor};`);
                ctx.js(`if (${rv} !== undefined) {`);
                field.value.codegenValidator(ctx, keyPath, rv);
                ctx.js(`}`);
            }
            else {
                ctx.js(`var ${rv} = ${r}${accessor};`);
                if (!(0, util_2.canSkipObjectKeyUndefinedCheck)(field.value.getSchema().__t)) {
                    const err = ctx.err(constants_1.ValidationError.KEY, [...path, field.key]);
                    ctx.js(`if (${rv} === undefined) return ${err};`);
                }
                field.value.codegenValidator(ctx, keyPath, `${r}${accessor}`);
            }
        }
        ctx.emitCustomValidators(this, path, r);
    }
    codegenJsonTextEncoder(ctx, value) {
        const { schema, fields } = this;
        const codegen = ctx.codegen;
        const r = codegen.getRegister();
        ctx.js(`var ${r} = ${value.use()};`);
        const rKeys = ctx.codegen.getRegister();
        if (schema.encodeUnknownFields) {
            ctx.js(`var ${rKeys} = new Set(Object.keys(${r}));`);
        }
        const requiredFields = fields.filter((field) => !(field instanceof ObjectOptionalFieldType));
        const optionalFields = fields.filter((field) => field instanceof ObjectOptionalFieldType);
        ctx.writeText('{');
        for (let i = 0; i < requiredFields.length; i++) {
            const field = requiredFields[i];
            if (i)
                ctx.writeText(',');
            ctx.writeText(JSON.stringify(field.key) + ':');
            const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
            const valueExpression = new JsExpression_1.JsExpression(() => `${r}${accessor}`);
            if (schema.encodeUnknownFields)
                ctx.js(`${rKeys}.delete(${JSON.stringify(field.key)});`);
            field.value.codegenJsonTextEncoder(ctx, valueExpression);
        }
        const rHasFields = codegen.getRegister();
        if (!requiredFields.length)
            ctx.js(`var ${rHasFields} = false;`);
        for (let i = 0; i < optionalFields.length; i++) {
            const field = optionalFields[i];
            const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
            const rValue = codegen.getRegister();
            if (schema.encodeUnknownFields)
                ctx.js(`${rKeys}.delete(${JSON.stringify(field.key)});`);
            ctx.js(`var ${rValue} = ${r}${accessor};`);
            ctx.js(`if (${rValue} !== undefined) {`);
            if (requiredFields.length) {
                ctx.writeText(',');
            }
            else {
                ctx.js(`if (${rHasFields}) s += ',';`);
                ctx.js(`${rHasFields} = true;`);
            }
            ctx.writeText(JSON.stringify(field.key) + ':');
            const valueExpression = new JsExpression_1.JsExpression(() => `${rValue}`);
            field.value.codegenJsonTextEncoder(ctx, valueExpression);
            ctx.js(`}`);
        }
        if (schema.encodeUnknownFields) {
            const [rList, ri, rLength, rk] = [codegen.r(), codegen.r(), codegen.r(), codegen.r()];
            ctx.js(`var ${rLength} = ${rKeys}.size;
if (${rLength}) {
  var ${rk}, ${rList} = Array.from(${rKeys}.values());
  for (var ${ri} = 0; ${ri} < ${rLength}; ${ri}++) {
    ${rk} = ${rList}[${ri}];
    s += ',' + asString(${rk}) + ':' + stringify(${r}[${rk}]);
  }
}`);
        }
        ctx.writeText('}');
    }
    codegenCborEncoder(ctx, value) {
        const codegen = ctx.codegen;
        const r = codegen.r();
        const fields = this.fields;
        const length = fields.length;
        const requiredFields = fields.filter((field) => !(field instanceof ObjectOptionalFieldType));
        const optionalFields = fields.filter((field) => field instanceof ObjectOptionalFieldType);
        const requiredLength = requiredFields.length;
        const optionalLength = optionalFields.length;
        const encodeUnknownFields = !!this.schema.encodeUnknownFields;
        const emitRequiredFields = () => {
            for (let i = 0; i < requiredLength; i++) {
                const field = requiredFields[i];
                ctx.blob(ctx.gen((encoder) => encoder.writeStr(field.key)));
                const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
                field.value.codegenCborEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
            }
        };
        const emitOptionalFields = () => {
            for (let i = 0; i < optionalLength; i++) {
                const field = optionalFields[i];
                const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
                codegen.js(`if (${r}${accessor} !== undefined) {`);
                ctx.blob(ctx.gen((encoder) => encoder.writeStr(field.key)));
                field.value.codegenCborEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
                codegen.js(`}`);
            }
        };
        const emitUnknownFields = () => {
            const rKeys = codegen.r();
            const rKey = codegen.r();
            const ri = codegen.r();
            const rLength = codegen.r();
            const keys = fields.map((field) => JSON.stringify(field.key));
            const rKnownFields = codegen.addConstant(`new Set([${keys.join(',')}])`);
            codegen.js(`var ${rKeys} = Object.keys(${r}), ${rLength} = ${rKeys}.length, ${rKey};`);
            codegen.js(`for (var ${ri} = 0; ${ri} < ${rLength}; ${ri}++) {`);
            codegen.js(`${rKey} = ${rKeys}[${ri}];`);
            codegen.js(`if (${rKnownFields}.has(${rKey})) continue;`);
            codegen.js(`encoder.writeStr(${rKey});`);
            codegen.js(`encoder.writeAny(${r}[${rKey}]);`);
            codegen.js(`}`);
        };
        ctx.js(`var ${r} = ${value.use()};`);
        if (!encodeUnknownFields && !optionalLength) {
            ctx.blob(ctx.gen((encoder) => encoder.writeObjHdr(length)));
            emitRequiredFields();
        }
        else if (!encodeUnknownFields) {
            ctx.blob(ctx.gen((encoder) => encoder.writeStartObj()));
            emitRequiredFields();
            emitOptionalFields();
            ctx.blob(ctx.gen((encoder) => encoder.writeEndObj()));
        }
        else {
            ctx.blob(ctx.gen((encoder) => encoder.writeStartObj()));
            emitRequiredFields();
            emitOptionalFields();
            emitUnknownFields();
            ctx.blob(ctx.gen((encoder) => encoder.writeEndObj()));
        }
    }
    codegenMessagePackEncoder(ctx, value) {
        const codegen = ctx.codegen;
        const r = codegen.r();
        const fields = this.fields;
        const length = fields.length;
        const requiredFields = fields.filter((field) => !(field instanceof ObjectOptionalFieldType));
        const optionalFields = fields.filter((field) => field instanceof ObjectOptionalFieldType);
        const requiredLength = requiredFields.length;
        const optionalLength = optionalFields.length;
        const totalMaxKnownFields = requiredLength + optionalLength;
        if (totalMaxKnownFields > 0xffff)
            throw new Error('Too many fields');
        const encodeUnknownFields = !!this.schema.encodeUnknownFields;
        const rFieldCount = codegen.r();
        const emitRequiredFields = () => {
            for (let i = 0; i < requiredLength; i++) {
                const field = requiredFields[i];
                ctx.blob(ctx.gen((encoder) => encoder.writeStr(field.key)));
                const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
                field.value.codegenMessagePackEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
            }
        };
        const emitOptionalFields = () => {
            for (let i = 0; i < optionalLength; i++) {
                const field = optionalFields[i];
                const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
                codegen.if(`${r}${accessor} !== undefined`, () => {
                    codegen.js(`${rFieldCount}++;`);
                    ctx.blob(ctx.gen((encoder) => encoder.writeStr(field.key)));
                    field.value.codegenMessagePackEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
                });
            }
        };
        const emitUnknownFields = () => {
            const ri = codegen.r();
            const rKeys = codegen.r();
            const rKey = codegen.r();
            const rLength = codegen.r();
            const keys = fields.map((field) => JSON.stringify(field.key));
            const rKnownFields = codegen.addConstant(`new Set([${keys.join(',')}])`);
            codegen.js(`var ${rKeys} = Object.keys(${r}), ${rLength} = ${rKeys}.length, ${rKey};`);
            codegen.js(`for (var ${ri} = 0; ${ri} < ${rLength}; ${ri}++) {`);
            codegen.js(`${rKey} = ${rKeys}[${ri}];`);
            codegen.js(`if (${rKnownFields}.has(${rKey})) continue;`);
            codegen.js(`${rFieldCount}++;`);
            codegen.js(`encoder.writeStr(${rKey});`);
            codegen.js(`encoder.writeAny(${r}[${rKey}]);`);
            codegen.js(`}`);
        };
        ctx.js(`var ${r} = ${value.use()};`);
        if (!encodeUnknownFields && !optionalLength) {
            ctx.blob(ctx.gen((encoder) => encoder.writeObjHdr(length)));
            emitRequiredFields();
        }
        else if (!encodeUnknownFields) {
            codegen.js(`var ${rFieldCount} = ${requiredLength};`);
            const rHeaderPosition = codegen.var('writer.x');
            ctx.blob(ctx.gen((encoder) => encoder.writeObjHdr(0xffff)));
            emitRequiredFields();
            emitOptionalFields();
            codegen.js(`view.setUint16(${rHeaderPosition} + 1, ${rFieldCount});`);
        }
        else {
            codegen.js(`var ${rFieldCount} = ${requiredLength};`);
            const rHeaderPosition = codegen.var('writer.x');
            ctx.blob(ctx.gen((encoder) => encoder.writeObjHdr(0xffffffff)));
            emitRequiredFields();
            emitOptionalFields();
            emitUnknownFields();
            codegen.js(`view.setUint32(${rHeaderPosition} + 1, ${rFieldCount});`);
        }
    }
    codegenJsonEncoder(ctx, value) {
        const codegen = ctx.codegen;
        const r = codegen.var(value.use());
        const fields = this.fields;
        const requiredFields = fields.filter((field) => !(field instanceof ObjectOptionalFieldType));
        const optionalFields = fields.filter((field) => field instanceof ObjectOptionalFieldType);
        const requiredLength = requiredFields.length;
        const optionalLength = optionalFields.length;
        const encodeUnknownFields = !!this.schema.encodeUnknownFields;
        const separatorBlob = ctx.gen((encoder) => encoder.writeObjSeparator());
        const keySeparatorBlob = ctx.gen((encoder) => encoder.writeObjKeySeparator());
        const endBlob = ctx.gen((encoder) => encoder.writeEndObj());
        const emitRequiredFields = () => {
            for (let i = 0; i < requiredLength; i++) {
                const field = requiredFields[i];
                ctx.blob(ctx.gen((encoder) => {
                    encoder.writeStr(field.key);
                    encoder.writeObjKeySeparator();
                }));
                const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
                field.value.codegenJsonEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
                ctx.blob(separatorBlob);
            }
        };
        const emitOptionalFields = () => {
            for (let i = 0; i < optionalLength; i++) {
                const field = optionalFields[i];
                const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
                codegen.if(`${r}${accessor} !== undefined`, () => {
                    ctx.blob(ctx.gen((encoder) => {
                        encoder.writeStr(field.key);
                    }));
                    ctx.blob(keySeparatorBlob);
                    field.value.codegenJsonEncoder(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
                    ctx.blob(separatorBlob);
                });
            }
        };
        const emitUnknownFields = () => {
            const rKeys = codegen.r();
            const rKey = codegen.r();
            const ri = codegen.r();
            const rLength = codegen.r();
            const keys = fields.map((field) => JSON.stringify(field.key));
            const rKnownFields = codegen.addConstant(`new Set([${keys.join(',')}])`);
            codegen.js(`var ${rKeys} = Object.keys(${r}), ${rLength} = ${rKeys}.length, ${rKey};`);
            codegen.js(`for (var ${ri} = 0; ${ri} < ${rLength}; ${ri}++) {`);
            codegen.js(`${rKey} = ${rKeys}[${ri}];`);
            codegen.js(`if (${rKnownFields}.has(${rKey})) continue;`);
            codegen.js(`encoder.writeStr(${rKey});`);
            ctx.blob(keySeparatorBlob);
            codegen.js(`encoder.writeAny(${r}[${rKey}]);`);
            ctx.blob(separatorBlob);
            codegen.js(`}`);
        };
        const emitEnding = () => {
            const rewriteLastSeparator = () => {
                for (let i = 0; i < endBlob.length; i++)
                    ctx.js(`uint8[writer.x - ${endBlob.length - i}] = ${endBlob[i]};`);
            };
            if (requiredFields.length) {
                rewriteLastSeparator();
            }
            else {
                codegen.if(`uint8[writer.x - 1] === ${separatorBlob[separatorBlob.length - 1]}`, () => {
                    rewriteLastSeparator();
                }, () => {
                    ctx.blob(endBlob);
                });
            }
        };
        ctx.blob(ctx.gen((encoder) => {
            encoder.writeStartObj();
        }));
        if (!encodeUnknownFields && !optionalLength) {
            emitRequiredFields();
            emitEnding();
        }
        else if (!encodeUnknownFields) {
            emitRequiredFields();
            emitOptionalFields();
            emitEnding();
        }
        else {
            emitRequiredFields();
            emitOptionalFields();
            emitUnknownFields();
            emitEnding();
        }
    }
    codegenCapacityEstimator(ctx, value) {
        const codegen = ctx.codegen;
        const r = codegen.var(value.use());
        const encodeUnknownFields = !!this.schema.encodeUnknownFields;
        if (encodeUnknownFields) {
            codegen.js(`size += maxEncodingCapacity(${r});`);
            return;
        }
        const fields = this.fields;
        const overhead = 5 + fields.length * 2;
        ctx.inc(overhead);
        for (const field of fields) {
            ctx.inc((0, json_size_1.maxEncodingCapacity)(field.key));
            const accessor = (0, normalizeAccessor_1.normalizeAccessor)(field.key);
            const isOptional = field instanceof ObjectOptionalFieldType;
            const block = () => field.value.codegenCapacityEstimator(ctx, new JsExpression_1.JsExpression(() => `${r}${accessor}`));
            if (isOptional) {
                codegen.if(`${r}${accessor} !== undefined`, block);
            }
            else
                block();
        }
    }
    random() {
        const schema = this.schema;
        const obj = schema.unknownFields ? json_random_1.RandomJson.genObject() : {};
        for (const field of this.fields) {
            if (field instanceof ObjectOptionalFieldType)
                if (Math.random() > 0.5)
                    continue;
            obj[field.key] = field.value.random();
        }
        return obj;
    }
    toTypeScriptAst() {
        const node = {
            node: 'TypeLiteral',
            members: [],
        };
        const fields = this.fields;
        for (const field of fields) {
            const member = {
                node: 'PropertySignature',
                name: field.key,
                type: field.value.toTypeScriptAst(),
            };
            if (field instanceof ObjectOptionalFieldType)
                member.optional = true;
            augmentWithComment(field.getSchema(), member);
            node.members.push(member);
        }
        if (this.schema.unknownFields || this.schema.encodeUnknownFields)
            node.members.push({
                node: 'IndexSignature',
                type: { node: 'UnknownKeyword' },
            });
        augmentWithComment(this.schema, node);
        return node;
    }
    toJson(value, system = this.system) {
        const fields = this.fields;
        const length = fields.length;
        if (!length)
            return '{}';
        const last = length - 1;
        let str = '{';
        for (let i = 0; i < last; i++) {
            const field = fields[i];
            const key = field.key;
            const fieldType = field.value;
            const val = value[key];
            if (val === undefined)
                continue;
            str += (0, asString_1.asString)(key) + ':' + fieldType.toJson(val, system) + ',';
        }
        const key = fields[last].key;
        const val = value[key];
        if (val !== undefined) {
            str += (0, asString_1.asString)(key) + ':' + fields[last].value.toJson(val, system);
        }
        else if (str.length > 1)
            str = str.slice(0, -1);
        return (str + '}');
    }
    toString(tab = '') {
        const _a = this.getSchema(), { __t, fields } = _a, rest = tslib_1.__rest(_a, ["__t", "fields"]);
        return (super.toString(tab) +
            (0, printTree_1.printTree)(tab, this.fields.map((field) => (tab) => field.toString(tab))));
    }
}
exports.ObjectType = ObjectType;
class RefType extends AbstractType {
    constructor(ref) {
        super();
        this.schema = schema.s.Ref(ref);
    }
    getRef() {
        return this.schema.ref;
    }
    toJsonSchema(ctx) {
        const ref = this.schema.ref;
        if (ctx)
            ctx.mentionRef(ref);
        const jsonSchema = Object.assign({ $ref: `#/$defs/${ref}` }, super.toJsonSchema(ctx));
        return jsonSchema;
    }
    getOptions() {
        const _a = this.schema, { __t, ref } = _a, options = tslib_1.__rest(_a, ["__t", "ref"]);
        return options;
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'ref');
        const { ref } = schema;
        if (typeof ref !== 'string')
            throw new Error('REF_TYPE');
        if (!ref)
            throw new Error('REF_EMPTY');
    }
    codegenValidator(ctx, path, r) {
        const refErr = (errorRegister) => {
            switch (ctx.options.errors) {
                case 'boolean':
                    return errorRegister;
                case 'string': {
                    return ctx.err(constants_1.ValidationError.REF, [...path, { r: errorRegister }]);
                }
                case 'object':
                default: {
                    return ctx.err(constants_1.ValidationError.REF, [...path], { refId: this.schema.ref, refError: errorRegister });
                }
            }
        };
        const system = ctx.options.system || this.system;
        if (!system)
            throw new Error('NO_SYSTEM');
        const validator = system.resolve(this.schema.ref).type.validator(ctx.options.errors);
        const d = ctx.codegen.linkDependency(validator);
        const rerr = ctx.codegen.getRegister();
        ctx.js(`var ${rerr} = ${d}(${r});`);
        ctx.js(`if (${rerr}) return ${refErr(rerr)};`);
    }
    codegenJsonTextEncoder(ctx, value) {
        const system = ctx.options.system || this.system;
        if (!system)
            throw new Error('NO_SYSTEM');
        const encoder = system.resolve(this.schema.ref).type.jsonTextEncoder();
        const d = ctx.codegen.linkDependency(encoder);
        ctx.js(`s += ${d}(${value.use()});`);
    }
    codegenBinaryEncoder(ctx, value) {
        const system = ctx.options.system || this.system;
        if (!system)
            throw new Error('NO_SYSTEM');
        const kind = ctx instanceof CborEncoderCodegenContext_1.CborEncoderCodegenContext
            ? 0
            : ctx instanceof MessagePackEncoderCodegenContext_1.MessagePackEncoderCodegenContext
                ? 1
                : 2;
        const targetType = system.resolve(this.schema.ref).type;
        switch (targetType.getTypeName()) {
            case 'str':
            case 'bool':
            case 'num':
            case 'any':
            case 'tup': {
                if (ctx instanceof CborEncoderCodegenContext_1.CborEncoderCodegenContext)
                    targetType.codegenCborEncoder(ctx, value);
                else if (ctx instanceof MessagePackEncoderCodegenContext_1.MessagePackEncoderCodegenContext)
                    targetType.codegenMessagePackEncoder(ctx, value);
                else if (ctx instanceof JsonEncoderCodegenContext_1.JsonEncoderCodegenContext)
                    targetType.codegenJsonEncoder(ctx, value);
                break;
            }
            default: {
                const encoder = targetType.encoder(kind);
                const d = ctx.codegen.linkDependency(encoder);
                ctx.js(`${d}(${value.use()}, encoder);`);
            }
        }
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        const system = ctx.options.system || this.system;
        if (!system)
            throw new Error('NO_SYSTEM');
        const estimator = system.resolve(this.schema.ref).type.capacityEstimator();
        const d = ctx.codegen.linkDependency(estimator);
        ctx.codegen.js(`size += ${d}(${value.use()});`);
    }
    random() {
        if (!this.system)
            throw new Error('NO_SYSTEM');
        const alias = this.system.resolve(this.schema.ref);
        return alias.type.random();
    }
    toTypeScriptAst() {
        return {
            node: 'GenericTypeAnnotation',
            id: {
                node: 'Identifier',
                name: this.schema.ref,
            },
        };
    }
    toJson(value, system = this.system) {
        if (!system)
            return 'null';
        const alias = system.resolve(this.schema.ref);
        return alias.type.toJson(value, system);
    }
    toStringTitle(tab = '') {
        const options = this.toStringOptions();
        return `${super.toStringTitle()} → [${this.schema.ref}]` + (options ? ` ${options}` : '');
    }
}
exports.RefType = RefType;
class OrType extends AbstractType {
    constructor(types, options) {
        super();
        this.types = types;
        this.__discriminator = undefined;
        this.schema = Object.assign(Object.assign({}, schema.s.Or()), options);
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { types: this.types.map((type) => type.getSchema()) });
    }
    toJsonSchema(ctx) {
        return {
            anyOf: this.types.map((type) => type.toJsonSchema(ctx)),
        };
    }
    getOptions() {
        const _a = this.schema, { __t, types } = _a, options = tslib_1.__rest(_a, ["__t", "types"]);
        return options;
    }
    options(options) {
        Object.assign(this.schema, options);
        return this;
    }
    discriminator() {
        if (this.__discriminator)
            return this.__discriminator;
        const expr = this.schema.discriminator;
        if (!expr || (expr[0] === 'num' && expr[1] === 0))
            throw new Error('NO_DISCRIMINATOR');
        const codegen = new json_expression_1.JsonExpressionCodegen({
            expression: expr,
            operators: operators_1.operatorsMap,
        });
        const fn = codegen.run().compile();
        return (this.__discriminator = (data) => +fn({ vars: new Vars_1.Vars(data) }));
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'or');
        const { types, discriminator } = schema;
        if (!discriminator || (discriminator[0] === 'num' && discriminator[1] === -1))
            throw new Error('DISCRIMINATOR');
        if (!Array.isArray(types))
            throw new Error('TYPES_TYPE');
        if (!types.length)
            throw new Error('TYPES_LENGTH');
        for (const type of this.types)
            type.validateSchema();
    }
    codegenValidator(ctx, path, r) {
        const types = this.types;
        const codegen = ctx.codegen;
        const length = types.length;
        if (length === 1) {
            types[0].codegenValidator(ctx, path, r);
            return;
        }
        const discriminator = this.discriminator();
        const d = codegen.linkDependency(discriminator);
        codegen.switch(`${d}(${r})`, types.map((type, index) => [
            index,
            () => {
                type.codegenValidator(ctx, path, r);
            },
        ]), () => {
            const err = ctx.err(constants_1.ValidationError.OR, path);
            ctx.js(`return ${err}`);
        });
    }
    codegenJsonTextEncoder(ctx, value) {
        ctx.js(`s += stringify(${value.use()});`);
    }
    codegenBinaryEncoder(ctx, value) {
        const codegen = ctx.codegen;
        const discriminator = this.discriminator();
        const d = codegen.linkDependency(discriminator);
        const types = this.types;
        codegen.switch(`${d}(${value.use()})`, types.map((type, index) => [
            index,
            () => {
                if (ctx instanceof CborEncoderCodegenContext_1.CborEncoderCodegenContext)
                    type.codegenCborEncoder(ctx, value);
                else if (ctx instanceof MessagePackEncoderCodegenContext_1.MessagePackEncoderCodegenContext)
                    type.codegenMessagePackEncoder(ctx, value);
                else if (ctx instanceof JsonEncoderCodegenContext_1.JsonEncoderCodegenContext)
                    type.codegenJsonEncoder(ctx, value);
            },
        ]));
    }
    codegenCborEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenMessagePackEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenJsonEncoder(ctx, value) {
        this.codegenBinaryEncoder(ctx, value);
    }
    codegenCapacityEstimator(ctx, value) {
        const codegen = ctx.codegen;
        const discriminator = this.discriminator();
        const d = codegen.linkDependency(discriminator);
        const types = this.types;
        codegen.switch(`${d}(${value.use()})`, types.map((type, index) => [
            index,
            () => {
                type.codegenCapacityEstimator(ctx, value);
            },
        ]));
    }
    random() {
        const types = this.types;
        const index = Math.floor(Math.random() * types.length);
        return types[index].random();
    }
    toTypeScriptAst() {
        const node = {
            node: 'UnionType',
            types: this.types.map((t) => t.toTypeScriptAst()),
        };
        return node;
    }
    toJson(value, system = this.system) {
        return JSON.stringify(value);
    }
    toString(tab = '') {
        return super.toString(tab) + (0, printTree_1.printTree)(tab, [...this.types.map((type) => (tab) => type.toString(tab))]);
    }
}
exports.OrType = OrType;
const fnNotImplemented = () => tslib_1.__awaiter(void 0, void 0, void 0, function* () {
    throw new Error('NOT_IMPLEMENTED');
});
class FunctionType extends AbstractType {
    constructor(req, res, options) {
        super();
        this.req = req;
        this.res = res;
        this.fn = fnNotImplemented;
        this.singleton = undefined;
        this.schema = Object.assign(Object.assign({}, options), schema.s.Function(schema.s.any, schema.s.any));
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { req: this.req.getSchema(), res: this.res.getSchema() });
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'fn');
        this.req.validateSchema();
        this.res.validateSchema();
    }
    random() {
        return () => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.res.random(); });
    }
    implement(singleton) {
        this.singleton = singleton;
        return this;
    }
    toTypeScriptAst() {
        throw new Error('Method not implemented.');
    }
    toString(tab = '') {
        return (super.toString(tab) +
            (0, printTree_1.printTree)(tab, [(tab) => 'req: ' + this.req.toString(tab), (tab) => 'res: ' + this.res.toString(tab)]));
    }
}
exports.FunctionType = FunctionType;
class FunctionStreamingType extends AbstractType {
    constructor(req, res, options) {
        super();
        this.req = req;
        this.res = res;
        this.isStreaming = true;
        this.singleton = undefined;
        this.schema = Object.assign(Object.assign({}, options), schema.s.Function$(schema.s.any, schema.s.any));
    }
    getSchema() {
        return Object.assign(Object.assign({}, this.schema), { req: this.req.getSchema(), res: this.res.getSchema() });
    }
    validateSchema() {
        const schema = this.getSchema();
        (0, validate_1.validateTType)(schema, 'fn$');
        this.req.validateSchema();
        this.res.validateSchema();
    }
    random() {
        return () => tslib_1.__awaiter(this, void 0, void 0, function* () { return this.res.random(); });
    }
    implement(singleton) {
        this.singleton = singleton;
        return this;
    }
    toTypeScriptAst() {
        throw new Error('Method not implemented.');
    }
    toString(tab = '') {
        return (super.toString(tab) +
            (0, printTree_1.printTree)(tab, [(tab) => 'req: ' + this.req.toString(tab), (tab) => 'res: ' + this.res.toString(tab)]));
    }
}
exports.FunctionStreamingType = FunctionStreamingType;
