"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Decoder = void 0;
const ClockDecoder_1 = require("../../../../json-crdt-patch/codec/clock/ClockDecoder");
const CrdtReader_1 = require("../../../../json-crdt-patch/util/binary/CrdtReader");
const clock_1 = require("../../../../json-crdt-patch/clock");
const Model_1 = require("../../../model/Model");
const CborDecoderBase_1 = require("../../../../json-pack/cbor/CborDecoderBase");
const nodes_1 = require("../../../nodes");
class Decoder extends CborDecoderBase_1.CborDecoderBase {
    constructor() {
        super(new CrdtReader_1.CrdtReader());
        this.time = -1;
        this.cStrChunk = () => {
            const reader = this.reader;
            const id = this.ts();
            const isTombstone = reader.uint8[reader.x] === 0;
            if (isTombstone) {
                reader.x++;
                const length = reader.vu39();
                return new nodes_1.StrChunk(id, length, '');
            }
            const text = this.readAsStr();
            return new nodes_1.StrChunk(id, text.length, text);
        };
        this.cBinChunk = () => {
            const reader = this.reader;
            const [deleted, length] = reader.b1vu28();
            const id = this.ts();
            if (deleted)
                return new nodes_1.BinChunk(id, length, undefined);
            else
                return new nodes_1.BinChunk(id, length, reader.buf(length));
        };
        this.cArrChunk = () => {
            const [deleted, length] = this.reader.b1vu28();
            const id = this.ts();
            if (deleted)
                return new nodes_1.ArrChunk(id, length, undefined);
            const ids = [];
            for (let i = 0; i < length; i++)
                ids.push(this.cNode().id);
            return new nodes_1.ArrChunk(id, length, ids);
        };
    }
    decode(data) {
        delete this.clockDecoder;
        this.time = -1;
        const reader = this.reader;
        reader.reset(data);
        const isServerTime = reader.u8() === 0;
        if (isServerTime) {
            const time = (this.time = reader.vu57());
            this.doc = Model_1.Model.withServerClock(time);
        }
        else {
            this.decodeClockTable();
            const clock = this.clockDecoder.clock;
            this.doc = Model_1.Model.withLogicalClock(clock);
        }
        this.doc.root = new nodes_1.RootNode(this.doc, this.cRoot().id);
        delete this.clockDecoder;
        return this.doc;
    }
    decodeClockTable() {
        const reader = this.reader;
        const clockTableOffset = reader.u32();
        const offset = reader.x;
        reader.x += clockTableOffset;
        const length = reader.vu39();
        const [sessionId, time] = reader.u53vu39();
        this.clockDecoder = new ClockDecoder_1.ClockDecoder(sessionId, time);
        for (let i = 1; i < length; i++) {
            const [sid, time] = reader.u53vu39();
            this.clockDecoder.pushTuple(sid, time);
        }
        reader.x = offset;
    }
    ts() {
        const decoderTime = this.time;
        const isLogical = decoderTime < 0;
        if (isLogical) {
            const [sessionIndex, timeDiff] = this.reader.id();
            return this.clockDecoder.decodeId(sessionIndex, timeDiff);
        }
        else {
            return new clock_1.Timestamp(1, this.reader.vu57());
        }
    }
    cRoot() {
        const reader = this.reader;
        const peek = reader.uint8[reader.x];
        return !peek ? Model_1.UNDEFINED : this.cNode();
    }
    cNode() {
        const reader = this.reader;
        const id = this.ts();
        const octet = reader.u8();
        const major = octet >> 5;
        const minor = octet & 0b11111;
        const length = minor < 24 ? minor : minor === 24 ? reader.u8() : minor === 25 ? reader.u16() : reader.u32();
        switch (major) {
            case 0:
                return this.cCon(id, length);
            case 1:
                return this.cVal(id);
            case 2:
                return this.cObj(id, length);
            case 3:
                return this.cVec(id, length);
            case 4:
                return this.cStr(id, length);
            case 5:
                return this.cBin(id, length);
            case 6:
                return this.cArr(id, length);
        }
        throw new Error('UNKNOWN_NODE');
    }
    cCon(id, length) {
        const doc = this.doc;
        const data = !length ? this.val() : this.ts();
        const node = new nodes_1.ConNode(id, data);
        doc.index.set(id, node);
        return node;
    }
    cVal(id) {
        const child = this.cNode();
        const doc = this.doc;
        const node = new nodes_1.ValNode(doc, id, child.id);
        doc.index.set(id, node);
        return node;
    }
    cObj(id, length) {
        const obj = new nodes_1.ObjNode(this.doc, id);
        for (let i = 0; i < length; i++)
            this.cObjChunk(obj);
        this.doc.index.set(id, obj);
        return obj;
    }
    cObjChunk(obj) {
        const key = this.key();
        obj.keys.set(key, this.cNode().id);
    }
    cVec(id, length) {
        const reader = this.reader;
        const obj = new nodes_1.VecNode(this.doc, id);
        const elements = obj.elements;
        for (let i = 0; i < length; i++) {
            const octet = reader.peak();
            if (!octet) {
                reader.x++;
                elements.push(undefined);
            }
            else
                elements.push(this.cNode().id);
        }
        this.doc.index.set(id, obj);
        return obj;
    }
    cStr(id, length) {
        const node = new nodes_1.StrNode(id);
        if (length)
            node.ingest(length, this.cStrChunk);
        this.doc.index.set(id, node);
        return node;
    }
    cBin(id, length) {
        const node = new nodes_1.BinNode(id);
        if (length)
            node.ingest(length, this.cBinChunk);
        this.doc.index.set(id, node);
        return node;
    }
    cArr(id, length) {
        const obj = new nodes_1.ArrNode(this.doc, id);
        if (length)
            obj.ingest(length, this.cArrChunk);
        this.doc.index.set(id, obj);
        return obj;
    }
}
exports.Decoder = Decoder;
