"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Encoder = void 0;
const tslib_1 = require("tslib");
const operations = tslib_1.__importStar(require("../../operations"));
const CrdtWriter_1 = require("../../util/binary/CrdtWriter");
const clock_1 = require("../../clock");
const CborEncoder_1 = require("../../../json-pack/cbor/CborEncoder");
class Encoder extends CborEncoder_1.CborEncoder {
    constructor(writer = new CrdtWriter_1.CrdtWriter()) {
        super(writer);
        this.writer = writer;
        this.patchSid = 0;
    }
    encode(patch) {
        this.writer.reset();
        const id = patch.getId();
        const sid = (this.patchSid = id.sid);
        const writer = this.writer;
        writer.vu57(sid);
        writer.vu57(id.time);
        const meta = patch.meta;
        if (meta === undefined)
            this.writeUndef();
        else
            this.writeArr([meta]);
        this.encodeOperations(patch);
        return writer.flush();
    }
    encodeOperations(patch) {
        const ops = patch.ops;
        const length = ops.length;
        this.writer.vu57(length);
        for (let i = 0; i < length; i++)
            this.encodeOperation(ops[i]);
    }
    encodeId(id) {
        const sessionId = id.sid;
        const time = id.time;
        const writer = this.writer;
        if (sessionId === this.patchSid) {
            writer.b1vu56(0, time);
        }
        else {
            writer.b1vu56(1, time);
            writer.vu57(sessionId);
        }
    }
    encodeTss(span) {
        this.encodeId(span);
        this.writer.vu57(span.span);
    }
    writeInsStr(length, obj, ref, str) {
        const writer = this.writer;
        if (length <= 0b111) {
            writer.u8((length << 5) | 12);
        }
        else {
            writer.u8(12);
            writer.vu57(length);
        }
        this.encodeId(obj);
        this.encodeId(ref);
        return writer.utf8(str);
    }
    encodeOperation(op) {
        const writer = this.writer;
        const constructor = op.constructor;
        switch (constructor) {
            case operations.NewConOp: {
                const operation = op;
                const val = operation.val;
                if (val instanceof clock_1.Timestamp) {
                    writer.u8(32 | 0);
                    this.encodeId(val);
                }
                else {
                    writer.u8(0);
                    this.writeAny(val);
                }
                break;
            }
            case operations.NewValOp: {
                writer.u8(1);
                break;
            }
            case operations.NewObjOp: {
                writer.u8(2);
                break;
            }
            case operations.NewVecOp: {
                writer.u8(3);
                break;
            }
            case operations.NewStrOp: {
                writer.u8(4);
                break;
            }
            case operations.NewBinOp: {
                writer.u8(5);
                break;
            }
            case operations.NewArrOp: {
                writer.u8(6);
                break;
            }
            case operations.InsValOp: {
                const operation = op;
                writer.u8(9);
                this.encodeId(operation.obj);
                this.encodeId(operation.val);
                break;
            }
            case operations.InsObjOp: {
                const operation = op;
                const data = operation.data;
                const length = data.length;
                if (length <= 0b111) {
                    writer.u8((length << 5) | 10);
                }
                else {
                    writer.u8(10);
                    writer.vu57(length);
                }
                this.encodeId(operation.obj);
                for (let i = 0; i < length; i++) {
                    const tuple = data[i];
                    this.writeStr(tuple[0]);
                    this.encodeId(tuple[1]);
                }
                break;
            }
            case operations.InsVecOp: {
                const operation = op;
                const data = operation.data;
                const length = data.length;
                if (length <= 0b111) {
                    writer.u8((length << 5) | 11);
                }
                else {
                    writer.u8(11);
                    writer.vu57(length);
                }
                this.encodeId(operation.obj);
                for (let i = 0; i < length; i++) {
                    const tuple = data[i];
                    writer.u8(tuple[0]);
                    this.encodeId(tuple[1]);
                }
                break;
            }
            case operations.InsStrOp: {
                const operation = op;
                const obj = operation.obj;
                const ref = operation.ref;
                const str = operation.data;
                const len1 = str.length;
                writer.ensureCapacity(24 + len1 * 4);
                const x = writer.x;
                const len2 = this.writeInsStr(len1, obj, ref, str);
                if (len1 !== len2) {
                    writer.x = x;
                    this.writeInsStr(len2, obj, ref, str);
                }
                break;
            }
            case operations.InsBinOp: {
                const operation = op;
                const buf = operation.data;
                const length = buf.length;
                if (length <= 0b111) {
                    writer.u8((length << 5) | 13);
                }
                else {
                    writer.u8(13);
                    writer.vu57(length);
                }
                this.encodeId(operation.obj);
                this.encodeId(operation.ref);
                writer.buf(buf, length);
                break;
            }
            case operations.InsArrOp: {
                const operation = op;
                const elements = operation.data;
                const length = elements.length;
                if (length <= 0b111) {
                    writer.u8((length << 5) | 14);
                }
                else {
                    writer.u8(14);
                    writer.vu57(length);
                }
                this.encodeId(operation.obj);
                this.encodeId(operation.ref);
                for (let i = 0; i < length; i++)
                    this.encodeId(elements[i]);
                break;
            }
            case operations.DelOp: {
                const operation = op;
                const what = operation.what;
                const length = what.length;
                if (length <= 0b111) {
                    writer.u8((length << 5) | 16);
                }
                else {
                    writer.u8(16);
                    writer.vu57(length);
                }
                this.encodeId(operation.obj);
                for (let i = 0; i < length; i++)
                    this.encodeTss(what[i]);
                break;
            }
            case operations.NopOp: {
                const operation = op;
                const length = operation.len;
                if (length <= 0b111) {
                    writer.u8((length << 5) | 17);
                }
                else {
                    writer.u8(17);
                    writer.vu57(length);
                }
                break;
            }
            default: {
                throw new Error('UNKNOWN_OP');
            }
        }
    }
}
exports.Encoder = Encoder;
