"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PatchBuilder = void 0;
const operations_1 = require("./operations");
const clock_1 = require("./clock");
const isUint8Array_1 = require("../util/buffers/isUint8Array");
const Patch_1 = require("./Patch");
const constants_1 = require("./constants");
const Tuple_1 = require("./builder/Tuple");
const Konst_1 = require("./builder/Konst");
const DelayedValueBuilder_1 = require("./builder/DelayedValueBuilder");
const maybeConst = (x) => {
    switch (typeof x) {
        case 'number':
        case 'boolean':
            return true;
        default:
            return x === null;
    }
};
class PatchBuilder {
    constructor(clock) {
        this.clock = clock;
        this.patch = new Patch_1.Patch();
    }
    nextTime() {
        return this.patch.nextTime() || this.clock.time;
    }
    flush() {
        const patch = this.patch;
        this.patch = new Patch_1.Patch();
        return patch;
    }
    obj() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewObjOp(id));
        return id;
    }
    arr() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewArrOp(id));
        return id;
    }
    vec() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewVecOp(id));
        return id;
    }
    str() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewStrOp(id));
        return id;
    }
    bin() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewBinOp(id));
        return id;
    }
    const(value) {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewConOp(id, value));
        return id;
    }
    val() {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.NewValOp(id));
        return id;
    }
    root(val) {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.InsValOp(id, constants_1.ORIGIN, val));
        return id;
    }
    insObj(obj, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_TUPLES');
        const id = this.clock.tick(1);
        const op = new operations_1.InsObjOp(id, obj, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    insVec(obj, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_TUPLES');
        const id = this.clock.tick(1);
        const op = new operations_1.InsVecOp(id, obj, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    setVal(obj, val) {
        this.pad();
        const id = this.clock.tick(1);
        const op = new operations_1.InsValOp(id, obj, val);
        this.patch.ops.push(op);
        return id;
    }
    insStr(obj, ref, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_STRING');
        const id = this.clock.tick(1);
        const op = new operations_1.InsStrOp(id, obj, ref, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    insBin(obj, ref, data) {
        this.pad();
        if (!data.length)
            throw new Error('EMPTY_BINARY');
        const id = this.clock.tick(1);
        const op = new operations_1.InsBinOp(id, obj, ref, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    insArr(arr, ref, data) {
        this.pad();
        const id = this.clock.tick(1);
        const op = new operations_1.InsArrOp(id, arr, ref, data);
        const span = op.span();
        if (span > 1)
            this.clock.tick(span - 1);
        this.patch.ops.push(op);
        return id;
    }
    del(obj, what) {
        this.pad();
        const id = this.clock.tick(1);
        this.patch.ops.push(new operations_1.DelOp(id, obj, what));
        return id;
    }
    nop(span) {
        this.pad();
        const id = this.clock.tick(span);
        this.patch.ops.push(new operations_1.NopOp(id, span));
        return id;
    }
    jsonObj(obj) {
        const id = this.obj();
        const keys = Object.keys(obj);
        if (keys.length) {
            const tuples = [];
            for (const k of keys) {
                const value = obj[k];
                const valueId = value instanceof clock_1.Timestamp ? value : maybeConst(value) ? this.const(value) : this.json(value);
                tuples.push([k, valueId]);
            }
            this.insObj(id, tuples);
        }
        return id;
    }
    jsonArr(arr) {
        const id = this.arr();
        if (arr.length) {
            const values = [];
            for (const el of arr)
                values.push(this.json(el));
            this.insArr(id, id, values);
        }
        return id;
    }
    jsonStr(str) {
        const id = this.str();
        if (str)
            this.insStr(id, id, str);
        return id;
    }
    jsonBin(bin) {
        const id = this.bin();
        if (bin.length)
            this.insBin(id, id, bin);
        return id;
    }
    jsonVal(value) {
        const valId = this.val();
        const id = this.const(value);
        this.setVal(valId, id);
        return valId;
    }
    jsonVec(vector) {
        const id = this.vec();
        const length = vector.length;
        if (length) {
            const writes = [];
            for (let i = 0; i < length; i++)
                writes.push([i, this.constOrJson(vector[i])]);
            this.insVec(id, writes);
        }
        return id;
    }
    json(json) {
        if (json instanceof clock_1.Timestamp)
            return json;
        if (json === undefined)
            return this.const(json);
        if (json instanceof Array)
            return this.jsonArr(json);
        if ((0, isUint8Array_1.isUint8Array)(json))
            return this.jsonBin(json);
        if (json instanceof Tuple_1.VectorDelayedValue)
            return this.jsonVec(json.slots);
        if (json instanceof Konst_1.Konst)
            return this.const(json.val);
        if (json instanceof DelayedValueBuilder_1.NodeBuilder)
            return json.build(this);
        switch (typeof json) {
            case 'object':
                return json === null ? this.jsonVal(json) : this.jsonObj(json);
            case 'string':
                return this.jsonStr(json);
            case 'number':
            case 'boolean':
                return this.jsonVal(json);
        }
        throw new Error('INVALID_JSON');
    }
    constOrJson(value) {
        if (value instanceof clock_1.Timestamp)
            return value;
        return maybeConst(value) ? this.const(value) : this.json(value);
    }
    maybeConst(value) {
        return value instanceof clock_1.Timestamp ? value : this.const(value);
    }
    pad() {
        const nextTime = this.patch.nextTime();
        if (!nextTime)
            return;
        const drift = this.clock.time - nextTime;
        if (drift > 0) {
            const id = (0, clock_1.ts)(this.clock.sid, nextTime);
            const padding = new operations_1.NopOp(id, drift);
            this.patch.ops.push(padding);
        }
    }
}
exports.PatchBuilder = PatchBuilder;
