import { JsExpression } from '../codegen/util/JsExpression';
import { printTree } from '../print/printTree';
import { RouterCodegenCtx, RouterCodegenOpts } from './codegen';
import { ExactStep, RegexStep, UntilStep } from './steps';
import { RoutingTreeNode } from './tree';
export class Router {
    options;
    destinations = [];
    constructor(options = {}) {
        this.options = options;
    }
    add(route, data) {
        const destination = Destination.from(route, data, this.options.defaultUntil);
        this.destinations.push(destination);
    }
    addDestination(destination) {
        this.destinations.push(destination);
    }
    tree() {
        const tree = new RoutingTreeNode();
        for (const destination of this.destinations) {
            for (const route of destination.routes) {
                tree.add(route, 0, destination);
            }
        }
        return tree;
    }
    compile() {
        const ctx = new RouterCodegenCtx();
        const node = new RouterCodegenOpts(new JsExpression(() => 'str'), '0');
        const tree = this.tree();
        tree.codegen(ctx, node);
        return ctx.codegen.compile();
    }
    toString(tab = '') {
        return (`${this.constructor.name}` +
            printTree(tab, [
                (tab) => 'Destinations' +
                    printTree(tab, this.destinations.map((d, i) => (tab) => `[${i}]: ` + d.toString(tab + ' '))),
                () => '',
                (tab) => 'RoutingTree' + printTree(tab, [(tab) => this.tree().toString(tab)]),
            ]));
    }
}
export class Destination {
    routes;
    data;
    static from(def, data, defaultUntil) {
        const routes = typeof def === 'string' ? [Route.from(def, defaultUntil)] : def.map((r) => Route.from(r));
        return new Destination(routes, data);
    }
    match;
    constructor(routes, data) {
        this.routes = routes;
        this.data = data;
        this.match = new Match(data, []);
    }
    toString(tab = '') {
        return (`${this.constructor.name} ` +
            (this.routes.length === 1
                ? this.routes[0].toString(tab)
                : printTree(tab, this.routes.map((r) => (tab) => r.toString(tab)))));
    }
}
export class Route {
    steps;
    static from(str, defaultUntil = '/') {
        const tokens = [];
        const matches = str.match(/\{[^\}]*\}/g);
        let i = 0;
        for (const match of matches ?? []) {
            const index = str.indexOf(match);
            if (index > i)
                tokens.push(str.substring(i, index));
            tokens.push(match);
            i = index + match.length;
        }
        if (i < str.length)
            tokens.push(str.substring(i));
        const steps = [];
        const length = tokens.length;
        for (let i = 0; i < length; i++) {
            const token = tokens[i];
            const isParameter = token.startsWith('{') && token.endsWith('}');
            if (isParameter) {
                const content = token.substring(1, token.length - 1);
                const [name = '', regex = '', until = ''] = content.split(':');
                if (!regex || regex === '*') {
                    const next = tokens[i + 1];
                    steps.push(new UntilStep(name, until || (next ? next[0] : defaultUntil)));
                }
                else {
                    steps.push(new RegexStep(name, regex, until));
                }
            }
            else {
                steps.push(new ExactStep(token));
            }
        }
        return new Route(steps);
    }
    constructor(steps) {
        this.steps = steps;
    }
    toText() {
        let str = '';
        for (const step in this.steps)
            str += this.steps[step].toText();
        return str;
    }
    toString(tab = '') {
        return this.toText();
    }
}
export class Match {
    data;
    params;
    constructor(data, params) {
        this.data = data;
        this.params = params;
    }
}
