import { enableCors } from './util';
import { Router } from '../../../util/router';
import { RpcMessageBatchProcessor } from '../../common/rpc/RpcMessageBatchProcessor';
import { RpcError, RpcErrorCodes, RpcErrorType } from '../../common/rpc/caller/error';
import { ConnectionContext } from '../context';
import { RpcMessageCodecs } from '../../common/codec/RpcMessageCodecs';
import { Value } from '../../common/messages/Value';
import { RpcCodecs } from '../../common/codec/RpcCodecs';
import { RpcMessageStreamProcessor } from '../../common';
const HDR_BAD_REQUEST = Buffer.from('400 Bad Request', 'utf8');
const HDR_NOT_FOUND = Buffer.from('404 Not Found', 'utf8');
const HDR_INTERNAL_SERVER_ERROR = Buffer.from('500 Internal Server Error', 'utf8');
const ERR_NOT_FOUND = RpcError.fromCode(RpcErrorCodes.NOT_FOUND, 'Not Found');
const ERR_INTERNAL = RpcError.internal();
export class RpcApp {
    options;
    codecs;
    app;
    maxRequestBodySize;
    router = new Router();
    batchProcessor;
    constructor(options) {
        this.options = options;
        this.app = options.uws;
        this.maxRequestBodySize = options.maxRequestBodySize;
        this.codecs = new RpcCodecs(options.codecs, new RpcMessageCodecs());
        this.batchProcessor = new RpcMessageBatchProcessor({ caller: options.caller });
    }
    enableCors() {
        enableCors(this.options.uws);
    }
    routeRaw(method, path, handler) {
        method = method.toLowerCase();
        this.router.add(method + path, handler);
    }
    route(method, path, handler) {
        this.routeRaw(method, path, async (ctx) => {
            const result = await handler(ctx);
            const res = ctx.res;
            if (res.aborted)
                return;
            const codec = ctx.resCodec;
            const encoder = codec.encoder;
            const writer = encoder.writer;
            writer.reset();
            if (res instanceof Value) {
                if (res.type)
                    res.type.encoder(codec.format)(res.data, encoder);
                else
                    encoder.writeAny(res.data);
            }
            else {
                encoder.writeAny(result);
            }
            if (res.aborted)
                return;
            ctx.sendResponse(writer.flush());
        });
    }
    enableHttpPing(path = '/ping') {
        this.route('GET', path, async () => {
            return 'pong';
        });
        return this;
    }
    enableHttpRpc(path = '/rpc') {
        this.routeRaw('POST', path, async (ctx) => {
            try {
                const res = ctx.res;
                const bodyUint8 = await ctx.requestBody(this.maxRequestBodySize);
                if (res.aborted)
                    return;
                const messageCodec = ctx.msgCodec;
                const incomingMessages = messageCodec.decodeBatch(ctx.reqCodec, bodyUint8);
                const outgoingMessages = await this.batchProcessor.onBatch(incomingMessages, ctx);
                if (res.aborted)
                    return;
                const resCodec = ctx.resCodec;
                messageCodec.encodeBatch(resCodec, outgoingMessages);
                const buf = resCodec.encoder.writer.flush();
                if (res.aborted)
                    return;
                res.end(buf);
            }
            catch (err) {
                if (typeof err === 'object' && err)
                    if (err.message === 'Invalid JSON')
                        throw RpcError.invalidRequest();
                throw RpcError.from(err);
            }
        });
        return this;
    }
    enableWsRpc(path = '/rpc') {
        const maxBackpressure = 4 * 1024 * 1024;
        this.app.ws(path, {
            idleTimeout: 0,
            maxPayloadLength: 4 * 1024 * 1024,
            upgrade: (res, req, context) => {
                const secWebSocketKey = req.getHeader('sec-websocket-key');
                const secWebSocketProtocol = req.getHeader('sec-websocket-protocol');
                const secWebSocketExtensions = req.getHeader('sec-websocket-extensions');
                const ctx = ConnectionContext.fromReqRes(req, res, null, this);
                res.upgrade({ ctx }, secWebSocketKey, secWebSocketProtocol, secWebSocketExtensions, context);
            },
            open: (ws_) => {
                const ws = ws_;
                const ctx = ws.ctx;
                const resCodec = ctx.resCodec;
                const msgCodec = ctx.msgCodec;
                const encoder = resCodec.encoder;
                const isBinary = resCodec.format !== 2 || msgCodec.format === 0;
                ws.rpc = new RpcMessageStreamProcessor({
                    caller: this.options.caller,
                    send: (messages) => {
                        if (ws.getBufferedAmount() > maxBackpressure)
                            return;
                        const writer = encoder.writer;
                        writer.reset();
                        msgCodec.encodeBatch(resCodec, messages);
                        const encoded = writer.flush();
                        ws.send(encoded, isBinary, false);
                    },
                    bufferSize: 1,
                    bufferTime: 0,
                });
            },
            message: (ws_, buf, isBinary) => {
                const ws = ws_;
                const ctx = ws.ctx;
                const reqCodec = ctx.reqCodec;
                const msgCodec = ctx.msgCodec;
                const uint8 = new Uint8Array(buf);
                const rpc = ws.rpc;
                try {
                    const messages = msgCodec.decodeBatch(reqCodec, uint8);
                    rpc.onMessages(messages, ctx);
                }
                catch (error) {
                    rpc.sendNotification('.err', RpcError.value(RpcError.invalidRequest()));
                }
            },
            close: (ws_, code, message) => {
                const ws = ws_;
                ws.rpc.stop();
            },
        });
        return this;
    }
    startRouting() {
        const matcher = this.router.compile();
        const codecs = this.codecs;
        let responseCodec = codecs.value.json;
        this.app.any('/*', async (res, req) => {
            res.onAborted(() => {
                res.aborted = true;
            });
            const method = req.getMethod();
            const url = req.getUrl();
            try {
                const match = matcher(method + url);
                if (!match) {
                    res.cork(() => {
                        res.writeStatus(HDR_NOT_FOUND);
                        res.end(RpcErrorType.encode(responseCodec, ERR_NOT_FOUND));
                    });
                    return;
                }
                const handler = match.data;
                const params = match.params;
                const ctx = ConnectionContext.fromReqRes(req, res, params, this);
                responseCodec = ctx.resCodec;
                await handler(ctx);
            }
            catch (err) {
                if (err instanceof RpcError) {
                    const error = err;
                    res.cork(() => {
                        res.writeStatus(HDR_BAD_REQUEST);
                        res.end(RpcErrorType.encode(responseCodec, error));
                    });
                    return;
                }
                if (err instanceof Value && err.data instanceof RpcError) {
                    const error = err.data;
                    res.cork(() => {
                        res.writeStatus(HDR_BAD_REQUEST);
                        res.end(RpcErrorType.encode(responseCodec, error));
                    });
                    return;
                }
                console.error(err);
                res.cork(() => {
                    res.writeStatus(HDR_INTERNAL_SERVER_ERROR);
                    res.end(RpcErrorType.encode(responseCodec, ERR_INTERNAL));
                });
            }
        });
    }
}
