import { firstValueFrom, isObservable, Observable, Subject } from 'rxjs';
import * as msg from '../../messages';
import { subscribeCompleteObserver } from '../../util/subscribeCompleteObserver';
import { TimedQueue } from '../../util/TimedQueue';
import { Value } from '../../messages/Value';
export class StreamingRpcClient {
    id = 1;
    buffer;
    calls = new Map();
    constructor({ send, bufferSize = 100, bufferTime = 10 }) {
        this.buffer = new TimedQueue();
        this.buffer.itemLimit = bufferSize;
        this.buffer.timeLimit = bufferTime;
        this.buffer.onFlush = send;
    }
    getInflightCallCount() {
        return this.calls.size;
    }
    onMessages(messages) {
        const length = messages.length;
        for (let i = 0; i < length; i++)
            this.onMessage(messages[i]);
    }
    onMessage(message) {
        if (message instanceof msg.ResponseCompleteMessage)
            return this.onResponseComplete(message);
        else if (message instanceof msg.ResponseDataMessage)
            return this.onResponseData(message);
        else if (message instanceof msg.ResponseErrorMessage)
            return this.onResponseError(message);
        return this.onRequestUnsubscribe(message);
    }
    onResponseComplete(message) {
        const { id, value } = message;
        const call = this.calls.get(id);
        if (!call)
            return;
        call.resFinalized = true;
        const data = value ? value.data : undefined;
        if (data !== undefined)
            call.res$.next(data);
        call.res$.complete();
    }
    onResponseData(message) {
        const { id, value } = message;
        const call = this.calls.get(id);
        if (!call)
            return;
        call.res$.next(value.data);
    }
    onResponseError(message) {
        const { id, value } = message;
        const call = this.calls.get(id);
        if (!call)
            return;
        call.resFinalized = true;
        call.res$.error(value.data);
    }
    onRequestUnsubscribe(message) {
        const { id } = message;
        const call = this.calls.get(id);
        if (!call)
            return;
        call.req$.complete();
    }
    call$(method, data) {
        const id = this.id++;
        if (this.id >= 0xffff)
            this.id = 1;
        if (this.calls.has(id))
            return this.call$(method, data);
        const req$ = new Subject();
        const res$ = new Subject();
        let finalizedStreams = 0;
        const cleanup = () => {
            finalizedStreams++;
            if (finalizedStreams === 2)
                this.calls.delete(id);
        };
        res$.subscribe({ error: cleanup, complete: cleanup });
        const entry = { req$, res$ };
        this.calls.set(id, entry);
        if (isObservable(data)) {
            let firstMessageSent = false;
            subscribeCompleteObserver(req$, {
                next: (value) => {
                    const messageMethod = firstMessageSent ? '' : method;
                    firstMessageSent = true;
                    const message = new msg.RequestDataMessage(id, messageMethod, new Value(value, undefined));
                    this.buffer.push(message);
                },
                error: (error) => {
                    cleanup();
                    const messageMethod = firstMessageSent ? '' : method;
                    const message = new msg.RequestErrorMessage(id, messageMethod, new Value(error, undefined));
                    this.buffer.push(message);
                },
                complete: (value) => {
                    cleanup();
                    const messageMethod = firstMessageSent ? '' : method;
                    const message = new msg.RequestCompleteMessage(id, messageMethod, new Value(value, undefined));
                    this.buffer.push(message);
                },
            });
            data.subscribe(req$);
        }
        else {
            this.buffer.push(new msg.RequestCompleteMessage(id, method, new Value(data, undefined)));
            req$.complete();
            cleanup();
        }
        return new Observable((observer) => {
            res$.subscribe(observer);
            return () => {
                if (!entry.resFinalized)
                    this.buffer.push(new msg.ResponseUnsubscribeMessage(id));
                res$.complete();
            };
        });
    }
    async call(method, request) {
        return await firstValueFrom(this.call$(method, request));
    }
    notify(method, data) {
        const value = new Value(data, undefined);
        this.buffer.push(new msg.NotificationMessage(method, value));
    }
    stop(reason = 'STOP') {
        this.buffer.onFlush = (message) => { };
        for (const call of this.calls.values()) {
            call.req$.error(new Error(reason));
            call.req$.error(new Error(reason));
        }
        this.calls.clear();
    }
    disconnect() {
        this.stop('DISCONNECT');
    }
}
