import { firstValueFrom, ReplaySubject, timer } from 'rxjs';
import { filter, first, share, switchMap, takeUntil } from 'rxjs/operators';
import { StreamingRpcClient } from './client/StreamingRpcClient';
import { ApiRpcCaller } from './caller/ApiRpcCaller';
import { RpcDuplex } from '../rpc/RpcDuplex';
import { RpcMessageStreamProcessor } from './RpcMessageStreamProcessor';
import { PersistentChannel } from '../channel';
export class RpcPersistentClient {
    channel;
    rpc;
    rpc$ = new ReplaySubject(1);
    constructor(params) {
        const ping = params.ping ?? 15000;
        const codec = params.codec;
        const textEncoder = new TextEncoder();
        this.channel = new PersistentChannel(params.channel);
        this.channel.open$.pipe(filter((open) => open)).subscribe(() => {
            const close$ = this.channel.open$.pipe(filter((open) => !open));
            const duplex = new RpcDuplex({
                client: new StreamingRpcClient({
                    ...(params.client || {}),
                    send: (messages) => {
                        const encoded = codec.encode(messages);
                        this.channel.send$(encoded).subscribe();
                    },
                }),
                server: new RpcMessageStreamProcessor({
                    ...(params.server || {
                        caller: new ApiRpcCaller({
                            api: {},
                        }),
                        onNotification: () => { },
                    }),
                    send: (messages) => {
                        const encoded = codec.encode(messages);
                        this.channel.send$(encoded).subscribe();
                    },
                }),
            });
            this.channel.message$.pipe(takeUntil(close$)).subscribe((data) => {
                const encoded = typeof data === 'string' ? textEncoder.encode(data) : new Uint8Array(data);
                const messages = codec.decode(encoded);
                duplex.onMessages((messages instanceof Array ? messages : [messages]), {});
            });
            if (ping) {
                timer(ping, ping)
                    .pipe(takeUntil(close$))
                    .subscribe(() => {
                    duplex.notify('.ping', undefined);
                });
            }
            if (this.rpc)
                this.rpc.disconnect();
            this.rpc = duplex;
            this.rpc$.next(duplex);
        });
    }
    call$(method, data) {
        return this.rpc$.pipe(first(), switchMap((rpc) => rpc.call$(method, data)), share());
    }
    call(method, data) {
        return firstValueFrom(this.call$(method, data));
    }
    notify(method, data) {
        this.rpc$.subscribe((rpc) => rpc.notify(method, data));
    }
    start() {
        this.channel.start();
    }
    stop() {
        this.channel.stop();
        if (this.rpc)
            this.rpc.stop();
    }
}
