import * as msg from '../messages';
import { RpcError } from './caller';
import { validateId, validateMethod } from './validation';
export class RpcMessageBatchProcessor {
    caller;
    constructor({ caller }) {
        this.caller = caller;
    }
    async onBatch(list, ctx) {
        try {
            const promises = [];
            const length = list.length;
            for (let i = 0; i < length; i++) {
                const message = list[i];
                switch (message.constructor) {
                    case msg.NotificationMessage:
                        this.onNotification(message, ctx);
                        break;
                    case msg.RequestDataMessage:
                    case msg.RequestCompleteMessage:
                    case msg.RequestErrorMessage:
                        promises.push(this.onRequest(message, ctx));
                        break;
                }
            }
            const settled = await Promise.allSettled(promises);
            const result = [];
            const settledLength = settled.length;
            for (let i = 0; i < settledLength; i++) {
                const item = settled[i];
                result.push(item.status === 'fulfilled' ? item.value : item.reason);
            }
            return result;
        }
        catch (error) {
            const value = RpcError.internalErrorValue();
            return [new msg.ResponseErrorMessage(-1, value)];
        }
    }
    onNotification(message, ctx) {
        const method = message.method;
        validateMethod(method);
        this.caller.notification(method, message.value.data, ctx).catch((error) => { });
    }
    async onRequest(message, ctx) {
        const id = message.id;
        validateId(id);
        const method = message.method;
        validateMethod(method);
        try {
            const value = message.value;
            const data = value ? value.data : undefined;
            const result = await this.caller.call(method, data, ctx);
            return new msg.ResponseCompleteMessage(id, result);
        }
        catch (error) {
            throw new msg.ResponseErrorMessage(id, error);
        }
    }
}
