import { RpcError } from '../../rpc/caller/error';
import { Value } from '../../messages/Value';
import * as msg from '../../messages';
import * as schema from './schema';
export class JsonRpc2RpcMessageCodec {
    id = 'json2.verbose';
    format = 2;
    encodeMessage(jsonCodec, message) {
        if (message instanceof msg.ResponseCompleteMessage || message instanceof msg.ResponseDataMessage) {
            const pojo = {
                id: message.id,
                result: message.value,
            };
            schema.JsonRpc2Response.encoder(jsonCodec.format)(pojo, jsonCodec.encoder);
        }
        else if (message instanceof msg.ResponseErrorMessage) {
            const error = message.value.data;
            if (error instanceof RpcError) {
                const pojo = {
                    id: message.id,
                    error: {
                        message: error.message,
                        code: error.errno,
                        data: error.toJson(),
                    },
                };
                schema.JsonRpc2Error.encoder(jsonCodec.format)(pojo, jsonCodec.encoder);
            }
            else {
                const pojo = {
                    id: message.id,
                    error: {
                        message: 'Unknown error',
                        code: 0,
                        data: error,
                    },
                };
                schema.JsonRpc2Error.encoder(jsonCodec.format)(pojo, jsonCodec.encoder);
            }
        }
        else if (message instanceof msg.NotificationMessage) {
            const pojo = {
                method: message.method,
                params: message.value,
            };
            schema.JsonRpc2Notification.encoder(jsonCodec.format)(pojo, jsonCodec.encoder);
        }
        else if (message instanceof msg.RequestCompleteMessage ||
            message instanceof msg.RequestDataMessage ||
            message instanceof msg.RequestErrorMessage) {
            const pojo = {
                jsonrpc: '2.0',
                id: message.id,
                method: message.method,
                params: message.value,
            };
            schema.JsonRpc2Request.encoder(jsonCodec.format)(pojo, jsonCodec.encoder);
        }
    }
    encodeBatch(jsonCodec, batch) {
        const length = batch.length;
        if (length === 1) {
            this.encodeMessage(jsonCodec, batch[0]);
        }
        else {
            switch (jsonCodec.format) {
                case 0:
                case 1: {
                    const encoder = jsonCodec.encoder;
                    encoder.writeArrHdr(length);
                    for (let i = 0; i < length; i++) {
                        this.encodeMessage(jsonCodec, batch[i]);
                    }
                    break;
                }
                case 2: {
                    const encoder = jsonCodec.encoder;
                    encoder.writeStartArr();
                    const last = length - 1;
                    for (let i = 0; i < last; i++) {
                        this.encodeMessage(jsonCodec, batch[i]);
                        encoder.writeArrSeparator();
                    }
                    if (length > 0)
                        this.encodeMessage(jsonCodec, batch[last]);
                    encoder.writeEndArr();
                    break;
                }
            }
        }
    }
    encode(jsonCodec, batch) {
        const encoder = jsonCodec.encoder;
        const writer = encoder.writer;
        writer.reset();
        this.encodeBatch(jsonCodec, batch);
        return writer.flush();
    }
    decodeBatch(jsonCodec, uint8) {
        try {
            let jsonRpcMessages = jsonCodec.decoder.read(uint8);
            if (!Array.isArray(jsonRpcMessages))
                jsonRpcMessages = [jsonRpcMessages];
            const messages = [];
            const length = jsonRpcMessages.length;
            for (let i = 0; i < length; i++)
                messages.push(this.fromJson(jsonRpcMessages[i]));
            return messages;
        }
        catch (error) {
            if (error instanceof RpcError)
                throw error;
            throw RpcError.invalidRequest();
        }
    }
    fromJson(message) {
        if (!message || typeof message !== 'object')
            throw RpcError.invalidRequest();
        if (message.id === undefined) {
            const notification = message;
            const data = notification.params;
            const value = new Value(data, undefined);
            return new msg.NotificationMessage(notification.method, value);
        }
        if (typeof message.method === 'string') {
            const request = message;
            const data = request.params;
            const value = data === undefined ? undefined : new Value(request.params, undefined);
            if (typeof request.id !== 'number')
                throw RpcError.invalidRequest();
            return new msg.RequestCompleteMessage(request.id, request.method, value);
        }
        if (message.result !== undefined) {
            const response = message;
            if (typeof response.id !== 'number')
                throw RpcError.invalidRequest();
            const data = response.result;
            const value = data === undefined ? undefined : new Value(response.result, undefined);
            return new msg.ResponseCompleteMessage(response.id, value);
        }
        if (message.error !== undefined) {
            const response = message;
            const value = new Value(response.error.data, undefined);
            if (typeof response.id !== 'number')
                throw RpcError.invalidRequest();
            return new msg.ResponseErrorMessage(response.id, value);
        }
        throw RpcError.invalidRequest();
    }
}
