import { RpcError, RpcErrorCodes } from '../../rpc/caller/error';
import * as msg from '../../messages';
import { Value } from '../../messages/Value';
import { CborEncoder } from '../../../../json-pack/cbor/CborEncoder';
import { MsgPackEncoder } from '../../../../json-pack/msgpack';
import { JsonEncoder } from '../../../../json-pack/json/JsonEncoder';
const fromJson = (arr) => {
    if (!(arr instanceof Array))
        throw RpcError.fromCode(RpcErrorCodes.BAD_REQUEST);
    const type = arr[0];
    switch (type) {
        case 1: {
            const data = arr[3];
            const value = data === undefined ? data : new Value(data, undefined);
            return new msg.RequestCompleteMessage(arr[1], arr[2], value);
        }
        case 0: {
            const data = arr[3];
            const value = data === undefined ? data : new Value(data, undefined);
            return new msg.RequestDataMessage(arr[1], arr[2], value);
        }
        case 2: {
            return new msg.RequestErrorMessage(arr[1], arr[2], new Value(arr[3], undefined));
        }
        case 3: {
            return new msg.RequestUnsubscribeMessage(arr[1]);
        }
        case 5: {
            const data = arr[2];
            const value = data === undefined ? data : new Value(data, undefined);
            return new msg.ResponseCompleteMessage(arr[1], value);
        }
        case 4: {
            return new msg.ResponseDataMessage(arr[1], new Value(arr[2], undefined));
        }
        case 6: {
            return new msg.ResponseErrorMessage(arr[1], new Value(arr[2], undefined));
        }
        case 7: {
            return new msg.ResponseUnsubscribeMessage(arr[1]);
        }
        case 8: {
            return new msg.NotificationMessage(arr[1], new Value(arr[2], undefined));
        }
    }
    throw RpcError.value(RpcError.validation('Unknown message type'));
};
export class CompactRpcMessageCodec {
    id = 'rx.compact';
    format = 1;
    encodeMessage(jsonCodec, message) {
        message.encodeCompact(jsonCodec);
    }
    encodeBatch(jsonCodec, batch) {
        const encoder = jsonCodec.encoder;
        if (encoder instanceof CborEncoder || encoder instanceof MsgPackEncoder) {
            const length = batch.length;
            encoder.writeArrHdr(length);
            for (let i = 0; i < length; i++)
                batch[i].encodeCompact(jsonCodec);
        }
        else if (encoder instanceof JsonEncoder) {
            const length = batch.length;
            const last = length - 1;
            encoder.writeStartArr();
            for (let i = 0; i < last; i++) {
                batch[i].encodeCompact(jsonCodec);
                encoder.writeArrSeparator();
            }
            if (length > 0)
                batch[last].encodeCompact(jsonCodec);
            encoder.writeEndArr();
        }
        else {
            const jsonMessages = [];
            const length = batch.length;
            for (let i = 0; i < length; i++)
                jsonMessages.push(batch[i].toCompact());
            const encoder = jsonCodec.encoder;
            encoder.writeArr(jsonMessages);
        }
    }
    encode(jsonCodec, batch) {
        const encoder = jsonCodec.encoder;
        const writer = encoder.writer;
        writer.reset();
        this.encodeBatch(jsonCodec, batch);
        return writer.flush();
    }
    decodeBatch(jsonCodec, uint8) {
        const decoder = jsonCodec.decoder;
        const value = decoder.read(uint8);
        if (!(value instanceof Array))
            throw RpcError.invalidRequest();
        if (typeof value[0] === 'number')
            return [fromJson(value)];
        const result = [];
        const length = value.length;
        for (let i = 0; i < length; i++) {
            const item = value[i];
            result.push(fromJson(item));
        }
        return result;
    }
    fromJson(compact) {
        return fromJson(compact);
    }
}
