import { TypeAlias } from './TypeAlias';
import { TypeBuilder } from '../type/TypeBuilder';
import { RefType } from '../type/classes';
import { printTree } from '../../util/print/printTree';
export class TypeSystem {
    t = new TypeBuilder(this);
    aliases = new Map();
    alias = (id, type) => {
        const existingAlias = this.aliases.get(id);
        if (existingAlias)
            return existingAlias;
        const alias = new TypeAlias(this, id, type);
        this.aliases.set(id, alias);
        return alias;
    };
    importTypes(types) {
        const result = {};
        for (const id in types)
            result[id] = this.alias(id, types[id]);
        return result;
    }
    unalias = (id) => {
        const alias = this.aliases.get(id);
        if (!alias)
            throw new Error(`Alias [id = ${id}] not found.`);
        return alias;
    };
    hasAlias = (id) => this.aliases.has(id);
    resolve = (id) => {
        const alias = this.unalias(id);
        return alias.type instanceof RefType ? this.resolve(alias.type.getRef()) : alias;
    };
    customValidators = new Map();
    addCustomValidator = (validator) => {
        if (this.customValidators.has(validator.name))
            throw new Error(`Validator [name = ${validator.name}] already exists.`);
        this.customValidators.set(validator.name, validator);
    };
    getCustomValidator = (name) => {
        const validator = this.customValidators.get(name);
        if (!validator)
            throw new Error(`Validator [name = ${name}] not found.`);
        return validator;
    };
    exportTypes() {
        const result = {};
        for (const [id, alias] of this.aliases.entries()) {
            result[id] = alias.getType().getSchema();
        }
        return result;
    }
    toString(tab = '') {
        const nl = () => '';
        return (this.constructor.name +
            printTree(tab, [
                (tab) => 'aliases' +
                    printTree(tab, [...this.aliases.values()].map((alias) => (tab) => alias.toString(tab))),
                this.customValidators.size ? nl : null,
                (tab) => 'validators' +
                    printTree(tab, [...this.customValidators.keys()].map((validator) => (tab) => `"${validator}"`)),
            ]));
    }
}
