import { parseArgs } from 'node:util';
import { TypeRouter } from '../json-type/system/TypeRouter';
import { TypeRouterCaller } from '../reactive-rpc/common/rpc/caller/TypeRouterCaller';
import { bufferToUint8Array } from '../util/buffers/bufferToUint8Array';
import { formatError } from './util';
import { defineBuiltinRoutes } from './methods';
import { defaultParams } from './defaultParams';
export class Cli {
    options;
    router;
    params;
    paramMap;
    types;
    t;
    caller;
    codecs;
    request;
    response;
    argv;
    stdout;
    stderr;
    stdin;
    exit;
    requestCodec;
    responseCodec;
    rawStdinInput;
    stdinInput;
    paramInstances = [];
    constructor(options) {
        this.options = options;
        let router = options.router ?? TypeRouter.create();
        router = defineBuiltinRoutes(router);
        this.router = router;
        this.params = options.params ?? defaultParams;
        this.paramMap = new Map();
        for (const param of this.params) {
            this.paramMap.set(param.param, param);
            if (param.short)
                this.paramMap.set(param.short, param);
        }
        this.caller = new TypeRouterCaller({ router, wrapInternalError: (err) => err });
        this.types = router.system;
        this.t = this.types.t;
        this.codecs = options.codecs;
        this.requestCodec = this.codecs.get(this.codecs.defaultCodec);
        this.responseCodec = this.codecs.get(this.codecs.defaultCodec);
        this.argv = options.argv ?? process.argv.slice(2);
        this.stdin = options.stdin ?? process.stdin;
        this.stdout = options.stdout ?? process.stdout;
        this.stderr = options.stderr ?? process.stderr;
        this.exit = options.exit ?? process.exit;
    }
    run() {
        this.runAsync();
    }
    param(param) {
        return this.paramMap.get(param);
    }
    async runAsync() {
        try {
            const args = parseArgs({
                args: this.argv,
                strict: false,
                allowPositionals: true,
            });
            for (let argKey of Object.keys(args.values)) {
                let pointer = '';
                const value = args.values[argKey];
                const slashIndex = argKey.indexOf('/');
                if (slashIndex !== -1) {
                    pointer = argKey.slice(slashIndex);
                    argKey = argKey.slice(0, slashIndex);
                }
                const param = this.param(argKey);
                if (!param) {
                    throw new Error(`Unknown parameter "${argKey}"`);
                }
                const instance = param.createInstance(this, pointer, value);
                this.paramInstances.push(instance);
                if (instance.onParam)
                    await instance.onParam();
            }
            const method = args.positionals[0];
            this.request = JSON.parse(args.positionals[1] || '{}');
            await this.readStdin();
            for (const instance of this.paramInstances)
                if (instance.onStdin)
                    await instance.onStdin();
            for (const instance of this.paramInstances)
                if (instance.onRequest)
                    await instance.onRequest();
            try {
                const ctx = { cli: this };
                for (const instance of this.paramInstances)
                    if (instance.onBeforeCall)
                        await instance.onBeforeCall(method, ctx);
                const value = await this.caller.call(method, this.request, ctx);
                this.response = value.data;
                for (const instance of this.paramInstances)
                    if (instance.onResponse)
                        await instance.onResponse();
                const buf = this.responseCodec.encode(this.response);
                this.stdout.write(buf);
            }
            catch (err) {
                const error = formatError(err);
                const buf = this.responseCodec.encode(error);
                this.stderr.write(buf);
                this.exit(1);
            }
        }
        catch (err) {
            const error = formatError(err);
            const buf = JSON.stringify(error, null, 4);
            this.stderr.write(buf);
            this.exit(1);
        }
    }
    cmd() {
        return this.options.cmd ?? '<cmd>';
    }
    async getStdin() {
        const stdin = this.stdin;
        if (stdin.isTTY)
            return Buffer.alloc(0);
        const result = [];
        let length = 0;
        for await (const chunk of stdin) {
            result.push(chunk);
            length += chunk.length;
        }
        return Buffer.concat(result, length);
    }
    async readStdin() {
        const stdin = this.stdin;
        const codec = this.requestCodec;
        if (stdin.isTTY)
            return Object.create(null);
        const input = await this.getStdin();
        if (codec.id === 'json') {
            const str = input.toString().trim();
            if (!str)
                return Object.create(null);
        }
        this.rawStdinInput = bufferToUint8Array(input);
        this.stdinInput = codec.decode(this.rawStdinInput);
    }
}
