import { AbstractOp } from './AbstractOp';
import { find, isObjectReference, isArrayReference, formatJsonPointer } from '../../json-pointer';
import { isTextNode, isElementNode } from '../util';
export class OpSplit extends AbstractOp {
    pos;
    props;
    constructor(path, pos, props) {
        super(path);
        this.pos = pos;
        this.props = props;
    }
    op() {
        return 'split';
    }
    code() {
        return 10;
    }
    apply(doc) {
        const ref = find(doc, this.path);
        if (ref.val === undefined)
            throw new Error('NOT_FOUND');
        const tuple = this.split(ref.val);
        if (isObjectReference(ref))
            ref.obj[ref.key] = tuple;
        else if (isArrayReference(ref)) {
            ref.obj[ref.key] = tuple[0];
            ref.obj.splice(ref.key + 1, 0, tuple[1]);
        }
        else
            doc = tuple;
        return { doc, old: ref.val };
    }
    split(node) {
        if (typeof node === 'string') {
            const { pos, props } = this;
            const before = node.slice(0, pos);
            const after = node.slice(pos);
            if (!props)
                return [before, after];
            const textNodes = [
                {
                    ...props,
                    text: before,
                },
                {
                    ...props,
                    text: after,
                },
            ];
            return textNodes;
        }
        else if (isTextNode(node)) {
            const { pos, props } = this;
            const before = node.text.slice(0, pos);
            const after = node.text.slice(pos);
            const textNodes = [
                {
                    ...node,
                    ...props,
                    text: before,
                },
                {
                    ...node,
                    ...props,
                    text: after,
                },
            ];
            return textNodes;
        }
        else if (isElementNode(node)) {
            const { pos, props } = this;
            const before = node.children.slice(0, pos);
            const after = node.children.slice(pos);
            const elementNodes = [
                {
                    ...node,
                    ...props,
                    children: before,
                },
                {
                    ...node,
                    ...props,
                    children: after,
                },
            ];
            return elementNodes;
        }
        else if (typeof node === 'number') {
            const { pos } = this;
            return [pos, node - pos];
        }
        else
            return [node, node];
    }
    toJson(parent) {
        const op = {
            op: 'split',
            path: formatJsonPointer(this.path),
            pos: this.pos,
        };
        if (this.props)
            op.props = this.props;
        return op;
    }
    toCompact(parent, verbose) {
        const opcode = verbose ? 'split' : 10;
        return this.props ? [opcode, this.path, this.pos, this.props] : [opcode, this.path, this.pos];
    }
    encode(encoder, parent) {
        encoder.encodeArrayHeader(this.props ? 4 : 3);
        encoder.writer.u8(10);
        encoder.encodeArray(this.path);
        encoder.encodeNumber(this.pos);
        if (this.props)
            encoder.encodeObject(this.props);
    }
}
