import * as util from './util';
import { Codegen } from '../util/codegen/Codegen';
import { Literal } from './codegen-steps';
import { createEvaluate } from './createEvaluate';
import { Vars } from './Vars';
export class JsonExpressionCodegen {
    options;
    codegen;
    evaluate;
    constructor(options) {
        this.options = options;
        this.codegen = new Codegen({
            args: ['ctx'],
            prologue: 'var vars = ctx.vars;',
            epilogue: '',
        });
        this.evaluate = createEvaluate({ ...options });
    }
    linkedOperandDeps = new Set();
    linkOperandDeps = (dependency, name) => {
        if (name) {
            if (this.linkedOperandDeps.has(name))
                return name;
            this.linkedOperandDeps.add(name);
        }
        else {
            name = this.codegen.getRegister();
        }
        this.codegen.linkDependency(dependency, name);
        return name;
    };
    operatorConst = (js) => {
        return this.codegen.addConstant(js);
    };
    subExpression = (expr) => {
        const codegen = new JsonExpressionCodegen({ ...this.options, expression: expr });
        const fn = codegen.run().compile();
        return fn;
    };
    onExpression(expr) {
        if (expr instanceof Array) {
            if (expr.length === 1)
                return new Literal(expr[0]);
        }
        else
            return new Literal(expr);
        const def = this.options.operators.get(expr[0]);
        if (def) {
            const [name, , arity, , codegen, impure] = def;
            util.assertArity(name, arity, expr);
            const operands = expr.slice(1).map((operand) => this.onExpression(operand));
            if (!impure) {
                const allLiterals = operands.every((expr) => expr instanceof Literal);
                if (allLiterals) {
                    const result = this.evaluate(expr, { vars: new Vars(undefined) });
                    return new Literal(result);
                }
            }
            const ctx = {
                expr,
                operands,
                createPattern: this.options.createPattern,
                operand: (operand) => this.onExpression(operand),
                link: this.linkOperandDeps,
                const: this.operatorConst,
                subExpression: this.subExpression,
            };
            return codegen(ctx);
        }
        return new Literal(false);
    }
    run() {
        const expr = this.onExpression(this.options.expression);
        this.codegen.js(`return ${expr};`);
        return this;
    }
    generate() {
        return this.codegen.generate();
    }
    compile() {
        const fn = this.codegen.compile();
        return (ctx) => {
            try {
                return fn(ctx);
            }
            catch (err) {
                if (err instanceof Error)
                    throw err;
                const error = new Error('Expression evaluation error.');
                error.value = err;
                throw error;
            }
        };
    }
}
