import * as operations from '../../json-crdt-patch/operations';
import { ConNode } from '../nodes/con/ConNode';
import { encoder, decoder } from '../codec/structural/binary/shared';
import { Timestamp, VectorClock, ServerVectorClock, compare, toDisplayString, } from '../../json-crdt-patch/clock';
import { ModelApi } from './api/ModelApi';
import { ORIGIN } from '../../json-crdt-patch/constants';
import { randomSessionId } from './util';
import { RootNode, ValNode, VecNode, ObjNode, StrNode, BinNode, ArrNode } from '../nodes';
import { printTree } from '../../util/print/printTree';
import { Extensions } from '../extensions/Extensions';
import { AvlMap } from '../../util/trees/avl/AvlMap';
export const UNDEFINED = new ConNode(ORIGIN, undefined);
export class Model {
    static withLogicalClock(clockOrSessionId) {
        const clock = typeof clockOrSessionId === 'number'
            ? new VectorClock(clockOrSessionId, 1)
            : clockOrSessionId || new VectorClock(randomSessionId(), 1);
        return new Model(clock);
    }
    static withServerClock(time = 0) {
        const clock = new ServerVectorClock(1, time);
        return new Model(clock);
    }
    static fromBinary(data) {
        return decoder.decode(data);
    }
    root = new RootNode(this, ORIGIN);
    clock;
    index = new AvlMap(compare);
    ext = new Extensions();
    constructor(clock) {
        this.clock = clock;
        if (!clock.time)
            clock.time = 1;
    }
    _api;
    get api() {
        if (!this._api)
            this._api = new ModelApi(this);
        return this._api;
    }
    get find() {
        return this.api.r.proxy();
    }
    tick = 0;
    onchange = undefined;
    applyBatch(patches) {
        const length = patches.length;
        for (let i = 0; i < length; i++)
            this.applyPatch(patches[i]);
    }
    applyPatch(patch) {
        const ops = patch.ops;
        const { length } = ops;
        for (let i = 0; i < length; i++)
            this.applyOperation(ops[i]);
        this.tick++;
        this.onchange?.();
    }
    applyOperation(op) {
        this.clock.observe(op.id, op.span());
        const index = this.index;
        if (op instanceof operations.InsStrOp) {
            const node = index.get(op.obj);
            if (node instanceof StrNode)
                node.ins(op.ref, op.id, op.data);
        }
        else if (op instanceof operations.NewObjOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new ObjNode(this, id));
        }
        else if (op instanceof operations.NewArrOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new ArrNode(this, id));
        }
        else if (op instanceof operations.NewStrOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new StrNode(id));
        }
        else if (op instanceof operations.NewValOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new ValNode(this, id, ORIGIN));
        }
        else if (op instanceof operations.NewConOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new ConNode(id, op.val));
        }
        else if (op instanceof operations.InsObjOp) {
            const node = index.get(op.obj);
            const tuples = op.data;
            const length = tuples.length;
            if (node instanceof ObjNode) {
                for (let i = 0; i < length; i++) {
                    const tuple = tuples[i];
                    const valueNode = index.get(tuple[1]);
                    if (!valueNode)
                        continue;
                    if (node.id.time >= tuple[1].time)
                        continue;
                    const old = node.put(tuple[0] + '', valueNode.id);
                    if (old)
                        this.deleteNodeTree(old);
                }
            }
        }
        else if (op instanceof operations.InsVecOp) {
            const node = index.get(op.obj);
            const tuples = op.data;
            const length = tuples.length;
            if (node instanceof VecNode) {
                for (let i = 0; i < length; i++) {
                    const tuple = tuples[i];
                    const valueNode = index.get(tuple[1]);
                    if (!valueNode)
                        continue;
                    if (node.id.time >= tuple[1].time)
                        continue;
                    const old = node.put(Number(tuple[0]), valueNode.id);
                    if (old)
                        this.deleteNodeTree(old);
                }
            }
        }
        else if (op instanceof operations.InsValOp) {
            const obj = op.obj;
            const node = obj.sid === 0 && obj.time === 0 ? this.root : index.get(obj);
            if (node instanceof ValNode) {
                const newValue = index.get(op.val);
                if (newValue) {
                    const old = node.set(op.val);
                    if (old)
                        this.deleteNodeTree(old);
                }
            }
        }
        else if (op instanceof operations.InsArrOp) {
            const node = index.get(op.obj);
            if (node instanceof ArrNode) {
                const nodes = [];
                const data = op.data;
                const length = data.length;
                for (let i = 0; i < length; i++) {
                    const stamp = data[i];
                    const valueNode = index.get(stamp);
                    if (!valueNode)
                        continue;
                    if (node.id.time >= stamp.time)
                        continue;
                    nodes.push(stamp);
                }
                if (nodes.length)
                    node.ins(op.ref, op.id, nodes);
            }
        }
        else if (op instanceof operations.DelOp) {
            const node = index.get(op.obj);
            if (node instanceof ArrNode) {
                const length = op.what.length;
                for (let i = 0; i < length; i++) {
                    const span = op.what[i];
                    for (let j = 0; j < span.span; j++) {
                        const id = node.getById(new Timestamp(span.sid, span.time + j));
                        if (id)
                            this.deleteNodeTree(id);
                    }
                }
                node.delete(op.what);
            }
            else if (node instanceof StrNode)
                node.delete(op.what);
            else if (node instanceof BinNode)
                node.delete(op.what);
        }
        else if (op instanceof operations.NewBinOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new BinNode(id));
        }
        else if (op instanceof operations.InsBinOp) {
            const node = index.get(op.obj);
            if (node instanceof BinNode)
                node.ins(op.ref, op.id, op.data);
        }
        else if (op instanceof operations.NewVecOp) {
            const id = op.id;
            if (!index.get(id))
                index.set(id, new VecNode(this, id));
        }
    }
    deleteNodeTree(value) {
        const isSystemNode = value.sid === 0;
        if (isSystemNode)
            return;
        const node = this.index.get(value);
        if (!node)
            return;
        node.children((child) => this.deleteNodeTree(child.id));
        this.index.del(value);
    }
    fork(sessionId = randomSessionId()) {
        const copy = Model.fromBinary(this.toBinary());
        if (copy.clock.sid !== sessionId && copy.clock instanceof VectorClock)
            copy.clock = copy.clock.fork(sessionId);
        copy.ext = this.ext;
        return copy;
    }
    clone() {
        return this.fork(this.clock.sid);
    }
    view() {
        return this.root.view();
    }
    toBinary() {
        return encoder.encode(this);
    }
    setSchema(schema) {
        if (this.clock.time < 2)
            this.api.root(schema);
        return this;
    }
    toString(tab = '') {
        const nl = () => '';
        const hasExtensions = this.ext.size() > 0;
        return (this.constructor.name +
            printTree(tab, [
                (tab) => this.root.toString(tab),
                nl,
                (tab) => {
                    const nodes = [];
                    this.index.forEach((item) => nodes.push(item.v));
                    return (`Index (${nodes.length} nodes)` +
                        (nodes.length
                            ? printTree(tab, nodes.map((node) => (tab) => `${node.constructor.name} ${toDisplayString(node.id)}`))
                            : ''));
                },
                nl,
                (tab) => `View${printTree(tab, [(tab) => String(JSON.stringify(this.view(), null, 2)).replace(/\n/g, '\n' + tab)])}`,
                nl,
                (tab) => this.clock.toString(tab),
                hasExtensions ? nl : null,
                hasExtensions ? (tab) => this.ext.toString(tab) : null,
            ]));
    }
}
