import { ClockDecoder } from '../../../../json-crdt-patch/codec/clock/ClockDecoder';
import { CrdtReader } from '../../../../json-crdt-patch/util/binary/CrdtReader';
import { Timestamp } from '../../../../json-crdt-patch/clock';
import { Model, UNDEFINED } from '../../../model/Model';
import { CborDecoderBase } from '../../../../json-pack/cbor/CborDecoderBase';
import { ArrNode, ArrChunk, BinNode, BinChunk, ConNode, ObjNode, RootNode, StrNode, StrChunk, ValNode, VecNode, } from '../../../nodes';
export class Decoder extends CborDecoderBase {
    doc;
    clockDecoder;
    time = -1;
    constructor() {
        super(new CrdtReader());
    }
    decode(data) {
        delete this.clockDecoder;
        this.time = -1;
        const reader = this.reader;
        reader.reset(data);
        const isServerTime = reader.u8() === 0;
        if (isServerTime) {
            const time = (this.time = reader.vu57());
            this.doc = Model.withServerClock(time);
        }
        else {
            this.decodeClockTable();
            const clock = this.clockDecoder.clock;
            this.doc = Model.withLogicalClock(clock);
        }
        this.doc.root = new RootNode(this.doc, this.cRoot().id);
        delete this.clockDecoder;
        return this.doc;
    }
    decodeClockTable() {
        const reader = this.reader;
        const clockTableOffset = reader.u32();
        const offset = reader.x;
        reader.x += clockTableOffset;
        const length = reader.vu39();
        const [sessionId, time] = reader.u53vu39();
        this.clockDecoder = new ClockDecoder(sessionId, time);
        for (let i = 1; i < length; i++) {
            const [sid, time] = reader.u53vu39();
            this.clockDecoder.pushTuple(sid, time);
        }
        reader.x = offset;
    }
    ts() {
        const decoderTime = this.time;
        const isLogical = decoderTime < 0;
        if (isLogical) {
            const [sessionIndex, timeDiff] = this.reader.id();
            return this.clockDecoder.decodeId(sessionIndex, timeDiff);
        }
        else {
            return new Timestamp(1, this.reader.vu57());
        }
    }
    cRoot() {
        const reader = this.reader;
        const peek = reader.uint8[reader.x];
        return !peek ? UNDEFINED : this.cNode();
    }
    cNode() {
        const reader = this.reader;
        const id = this.ts();
        const octet = reader.u8();
        const major = octet >> 5;
        const minor = octet & 0b11111;
        const length = minor < 24 ? minor : minor === 24 ? reader.u8() : minor === 25 ? reader.u16() : reader.u32();
        switch (major) {
            case 0:
                return this.cCon(id, length);
            case 1:
                return this.cVal(id);
            case 2:
                return this.cObj(id, length);
            case 3:
                return this.cVec(id, length);
            case 4:
                return this.cStr(id, length);
            case 5:
                return this.cBin(id, length);
            case 6:
                return this.cArr(id, length);
        }
        throw new Error('UNKNOWN_NODE');
    }
    cCon(id, length) {
        const doc = this.doc;
        const data = !length ? this.val() : this.ts();
        const node = new ConNode(id, data);
        doc.index.set(id, node);
        return node;
    }
    cVal(id) {
        const child = this.cNode();
        const doc = this.doc;
        const node = new ValNode(doc, id, child.id);
        doc.index.set(id, node);
        return node;
    }
    cObj(id, length) {
        const obj = new ObjNode(this.doc, id);
        for (let i = 0; i < length; i++)
            this.cObjChunk(obj);
        this.doc.index.set(id, obj);
        return obj;
    }
    cObjChunk(obj) {
        const key = this.key();
        obj.keys.set(key, this.cNode().id);
    }
    cVec(id, length) {
        const reader = this.reader;
        const obj = new VecNode(this.doc, id);
        const elements = obj.elements;
        for (let i = 0; i < length; i++) {
            const octet = reader.peak();
            if (!octet) {
                reader.x++;
                elements.push(undefined);
            }
            else
                elements.push(this.cNode().id);
        }
        this.doc.index.set(id, obj);
        return obj;
    }
    cStr(id, length) {
        const node = new StrNode(id);
        if (length)
            node.ingest(length, this.cStrChunk);
        this.doc.index.set(id, node);
        return node;
    }
    cStrChunk = () => {
        const reader = this.reader;
        const id = this.ts();
        const isTombstone = reader.uint8[reader.x] === 0;
        if (isTombstone) {
            reader.x++;
            const length = reader.vu39();
            return new StrChunk(id, length, '');
        }
        const text = this.readAsStr();
        return new StrChunk(id, text.length, text);
    };
    cBin(id, length) {
        const node = new BinNode(id);
        if (length)
            node.ingest(length, this.cBinChunk);
        this.doc.index.set(id, node);
        return node;
    }
    cBinChunk = () => {
        const reader = this.reader;
        const [deleted, length] = reader.b1vu28();
        const id = this.ts();
        if (deleted)
            return new BinChunk(id, length, undefined);
        else
            return new BinChunk(id, length, reader.buf(length));
    };
    cArr(id, length) {
        const obj = new ArrNode(this.doc, id);
        if (length)
            obj.ingest(length, this.cArrChunk);
        this.doc.index.set(id, obj);
        return obj;
    }
    cArrChunk = () => {
        const [deleted, length] = this.reader.b1vu28();
        const id = this.ts();
        if (deleted)
            return new ArrChunk(id, length, undefined);
        const ids = [];
        for (let i = 0; i < length; i++)
            ids.push(this.cNode().id);
        return new ArrChunk(id, length, ids);
    };
}
