import * as operations from '../../operations';
import { Timestamp } from '../../clock';
import { toBase64 } from '../../../util/base64/toBase64';
const timestamp = (sid, ts) => {
    const tsSessionId = ts.sid;
    return tsSessionId === sid ? ts.time : [tsSessionId, ts.time];
};
const timespan = (sid, span) => {
    const ts = timestamp(sid, span);
    if (ts instanceof Array) {
        ts.push(span.span);
        return ts;
    }
    return [ts, span.span];
};
export const encode = (patch) => {
    const id = patch.getId();
    if (!id)
        throw new Error('PATCH_EMPTY');
    const sid = id.sid;
    const time = id.time;
    const header = sid === 1 ? [time] : [[sid, time]];
    const meta = patch.meta;
    if (meta !== undefined)
        header.push(meta);
    const res = [header];
    for (const op of patch.ops) {
        if (op instanceof operations.NewConOp) {
            const val = op.val;
            if (val instanceof Timestamp) {
                res.push([0, timestamp(sid, val), true]);
            }
            else if (val === undefined) {
                res.push([0]);
            }
            else {
                res.push([0, val]);
            }
        }
        else if (op instanceof operations.NewValOp) {
            res.push([1]);
        }
        else if (op instanceof operations.NewObjOp) {
            res.push([2]);
        }
        else if (op instanceof operations.NewVecOp) {
            res.push([3]);
        }
        else if (op instanceof operations.NewStrOp) {
            res.push([4]);
        }
        else if (op instanceof operations.NewBinOp) {
            res.push([5]);
        }
        else if (op instanceof operations.NewArrOp) {
            res.push([6]);
        }
        else if (op instanceof operations.InsValOp) {
            res.push([9, timestamp(sid, op.obj), timestamp(sid, op.val)]);
        }
        else if (op instanceof operations.InsObjOp) {
            const tuples = [];
            for (const [key, value] of op.data)
                tuples.push([key, timestamp(sid, value)]);
            const operation = [
                10,
                timestamp(sid, op.obj),
                tuples,
            ];
            res.push(operation);
        }
        else if (op instanceof operations.InsVecOp) {
            const tuples = [];
            for (const [key, value] of op.data)
                tuples.push([key, timestamp(sid, value)]);
            const operation = [
                11,
                timestamp(sid, op.obj),
                tuples,
            ];
            res.push(operation);
        }
        else if (op instanceof operations.InsStrOp) {
            const operation = [
                12,
                timestamp(sid, op.obj),
                timestamp(sid, op.ref),
                op.data,
            ];
            res.push(operation);
        }
        else if (op instanceof operations.InsBinOp) {
            const operation = [
                13,
                timestamp(sid, op.obj),
                timestamp(sid, op.ref),
                toBase64(op.data),
            ];
            res.push(operation);
        }
        else if (op instanceof operations.InsArrOp) {
            const elements = [];
            for (const element of op.data)
                elements.push(timestamp(sid, element));
            const operation = [
                14,
                timestamp(sid, op.obj),
                timestamp(sid, op.ref),
                elements,
            ];
            res.push(operation);
        }
        else if (op instanceof operations.DelOp) {
            const operation = [
                16,
                timestamp(sid, op.obj),
                op.what.map((span) => timespan(sid, span)),
            ];
            res.push(operation);
        }
        else if (op instanceof operations.NopOp) {
            const operation = [17];
            const len = op.len;
            if (len > 1)
                operation.push(len);
            res.push(operation);
        }
    }
    return res;
};
