import { JsonPackExtension, JsonPackValue } from '../json-pack/msgpack';
import { fromBase64, toBase64 } from '../util/base64';
import { isUint8Array } from '../util/buffers/isUint8Array';
import { binUriStart, msgPackExtStart, msgPackUriStart } from './constants';
const binUriStartLength = binUriStart.length;
const msgPackUriStartLength = msgPackUriStart.length;
const msgPackExtStartLength = msgPackExtStart.length;
const minDataUri = Math.min(binUriStartLength, msgPackUriStartLength);
const parseExtDataUri = (uri) => {
    uri = uri.substring(msgPackExtStartLength);
    const commaIndex = uri.indexOf(',');
    if (commaIndex === -1)
        throw new Error('INVALID_EXT_DATA_URI');
    const typeString = uri.substring(0, commaIndex);
    const buf = fromBase64(uri.substring(commaIndex + 1));
    return new JsonPackExtension(Number(typeString), buf);
};
export const unwrapBinary = (value) => {
    if (!value)
        return value;
    if (value instanceof Array) {
        const len = value.length;
        for (let i = 0; i < len; i++) {
            const item = value[i];
            switch (typeof item) {
                case 'object': {
                    unwrapBinary(item);
                    continue;
                }
                case 'string': {
                    if (item.length < minDataUri)
                        continue;
                    if (item.substring(0, binUriStartLength) === binUriStart)
                        value[i] = fromBase64(item.substring(binUriStartLength));
                    else if (item.substring(0, msgPackUriStartLength) === msgPackUriStart)
                        value[i] = new JsonPackValue(fromBase64(item.substring(msgPackUriStartLength)));
                    else if (item.substring(0, msgPackExtStartLength) === msgPackExtStart)
                        value[i] = parseExtDataUri(item);
                }
            }
        }
        return value;
    }
    if (typeof value === 'object') {
        for (const key in value) {
            const item = value[key];
            switch (typeof item) {
                case 'object': {
                    unwrapBinary(item);
                    continue;
                }
                case 'string': {
                    if (item.length < minDataUri)
                        continue;
                    if (item.substring(0, binUriStartLength) === binUriStart) {
                        const buf = fromBase64(item.substring(binUriStartLength));
                        value[key] = buf;
                    }
                    else if (item.substring(0, msgPackUriStartLength) === msgPackUriStart) {
                        value[key] = new JsonPackValue(fromBase64(item.substring(msgPackUriStartLength)));
                    }
                    else if (item.substring(0, msgPackExtStartLength) === msgPackExtStart)
                        value[key] = parseExtDataUri(item);
                }
            }
        }
        return value;
    }
    if (typeof value === 'string') {
        if (value.length < minDataUri)
            return value;
        if (value.substring(0, binUriStartLength) === binUriStart)
            return fromBase64(value.substring(binUriStartLength));
        if (value.substring(0, msgPackUriStartLength) === msgPackUriStart)
            return new JsonPackValue(fromBase64(value.substring(msgPackUriStartLength)));
        if (value.substring(0, msgPackExtStartLength) === msgPackExtStart)
            return parseExtDataUri(value);
        else
            return value;
    }
    return value;
};
export const parse = (json) => {
    const parsed = JSON.parse(json);
    return unwrapBinary(parsed);
};
export const stringifyBinary = (value) => (binUriStart + toBase64(value));
export const wrapBinary = (value) => {
    if (!value)
        return value;
    if (isUint8Array(value))
        return stringifyBinary(value);
    if (value instanceof Array) {
        const out = [];
        const len = value.length;
        for (let i = 0; i < len; i++) {
            const item = value[i];
            out.push(!item || typeof item !== 'object' ? item : wrapBinary(item));
        }
        return out;
    }
    if (value instanceof JsonPackValue)
        return msgPackUriStart + toBase64(value.val);
    if (value instanceof JsonPackExtension)
        return msgPackExtStart + value.tag + ',' + toBase64(value.val);
    if (typeof value === 'object') {
        const out = {};
        for (const key in value) {
            const item = value[key];
            out[key] = !item || typeof item !== 'object' ? item : wrapBinary(item);
        }
        return out;
    }
    return value;
};
export const stringify = (value, replacer, space) => {
    const wrapped = wrapBinary(value);
    return JSON.stringify(wrapped, replacer, space);
};
