"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Cli = void 0;
const tslib_1 = require("tslib");
const node_util_1 = require("node:util");
const TypeRouter_1 = require("../json-type/system/TypeRouter");
const TypeRouterCaller_1 = require("../reactive-rpc/common/rpc/caller/TypeRouterCaller");
const bufferToUint8Array_1 = require("../util/buffers/bufferToUint8Array");
const util_1 = require("./util");
const methods_1 = require("./methods");
const defaultParams_1 = require("./defaultParams");
class Cli {
    constructor(options) {
        var _a, _b, _c, _d, _e, _f, _g;
        this.options = options;
        this.paramInstances = [];
        let router = (_a = options.router) !== null && _a !== void 0 ? _a : TypeRouter_1.TypeRouter.create();
        router = (0, methods_1.defineBuiltinRoutes)(router);
        this.router = router;
        this.params = (_b = options.params) !== null && _b !== void 0 ? _b : defaultParams_1.defaultParams;
        this.paramMap = new Map();
        for (const param of this.params) {
            this.paramMap.set(param.param, param);
            if (param.short)
                this.paramMap.set(param.short, param);
        }
        this.caller = new TypeRouterCaller_1.TypeRouterCaller({ router, wrapInternalError: (err) => err });
        this.types = router.system;
        this.t = this.types.t;
        this.codecs = options.codecs;
        this.requestCodec = this.codecs.get(this.codecs.defaultCodec);
        this.responseCodec = this.codecs.get(this.codecs.defaultCodec);
        this.argv = (_c = options.argv) !== null && _c !== void 0 ? _c : process.argv.slice(2);
        this.stdin = (_d = options.stdin) !== null && _d !== void 0 ? _d : process.stdin;
        this.stdout = (_e = options.stdout) !== null && _e !== void 0 ? _e : process.stdout;
        this.stderr = (_f = options.stderr) !== null && _f !== void 0 ? _f : process.stderr;
        this.exit = (_g = options.exit) !== null && _g !== void 0 ? _g : process.exit;
    }
    run() {
        this.runAsync();
    }
    param(param) {
        return this.paramMap.get(param);
    }
    runAsync() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            try {
                const args = (0, node_util_1.parseArgs)({
                    args: this.argv,
                    strict: false,
                    allowPositionals: true,
                });
                for (let argKey of Object.keys(args.values)) {
                    let pointer = '';
                    const value = args.values[argKey];
                    const slashIndex = argKey.indexOf('/');
                    if (slashIndex !== -1) {
                        pointer = argKey.slice(slashIndex);
                        argKey = argKey.slice(0, slashIndex);
                    }
                    const param = this.param(argKey);
                    if (!param) {
                        throw new Error(`Unknown parameter "${argKey}"`);
                    }
                    const instance = param.createInstance(this, pointer, value);
                    this.paramInstances.push(instance);
                    if (instance.onParam)
                        yield instance.onParam();
                }
                const method = args.positionals[0];
                this.request = JSON.parse(args.positionals[1] || '{}');
                yield this.readStdin();
                for (const instance of this.paramInstances)
                    if (instance.onStdin)
                        yield instance.onStdin();
                for (const instance of this.paramInstances)
                    if (instance.onRequest)
                        yield instance.onRequest();
                try {
                    const ctx = { cli: this };
                    for (const instance of this.paramInstances)
                        if (instance.onBeforeCall)
                            yield instance.onBeforeCall(method, ctx);
                    const value = yield this.caller.call(method, this.request, ctx);
                    this.response = value.data;
                    for (const instance of this.paramInstances)
                        if (instance.onResponse)
                            yield instance.onResponse();
                    const buf = this.responseCodec.encode(this.response);
                    this.stdout.write(buf);
                }
                catch (err) {
                    const error = (0, util_1.formatError)(err);
                    const buf = this.responseCodec.encode(error);
                    this.stderr.write(buf);
                    this.exit(1);
                }
            }
            catch (err) {
                const error = (0, util_1.formatError)(err);
                const buf = JSON.stringify(error, null, 4);
                this.stderr.write(buf);
                this.exit(1);
            }
        });
    }
    cmd() {
        var _a;
        return (_a = this.options.cmd) !== null && _a !== void 0 ? _a : '<cmd>';
    }
    getStdin() {
        var _a, e_1, _b, _c;
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const stdin = this.stdin;
            if (stdin.isTTY)
                return Buffer.alloc(0);
            const result = [];
            let length = 0;
            try {
                for (var _d = true, stdin_1 = tslib_1.__asyncValues(stdin), stdin_1_1; stdin_1_1 = yield stdin_1.next(), _a = stdin_1_1.done, !_a; _d = true) {
                    _c = stdin_1_1.value;
                    _d = false;
                    const chunk = _c;
                    result.push(chunk);
                    length += chunk.length;
                }
            }
            catch (e_1_1) { e_1 = { error: e_1_1 }; }
            finally {
                try {
                    if (!_d && !_a && (_b = stdin_1.return)) yield _b.call(stdin_1);
                }
                finally { if (e_1) throw e_1.error; }
            }
            return Buffer.concat(result, length);
        });
    }
    readStdin() {
        return tslib_1.__awaiter(this, void 0, void 0, function* () {
            const stdin = this.stdin;
            const codec = this.requestCodec;
            if (stdin.isTTY)
                return Object.create(null);
            const input = yield this.getStdin();
            if (codec.id === 'json') {
                const str = input.toString().trim();
                if (!str)
                    return Object.create(null);
            }
            this.rawStdinInput = (0, bufferToUint8Array_1.bufferToUint8Array)(input);
            this.stdinInput = codec.decode(this.rawStdinInput);
        });
    }
}
exports.Cli = Cli;
