"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Decoder = void 0;
const CrdtReader_1 = require("../../util/binary/CrdtReader");
const clock_1 = require("../../clock");
const PatchBuilder_1 = require("../../PatchBuilder");
const CborDecoder_1 = require("../../../json-pack/cbor/CborDecoder");
class Decoder extends CborDecoder_1.CborDecoder {
    constructor(reader = new CrdtReader_1.CrdtReader()) {
        super(reader);
    }
    decode(data) {
        const reader = this.reader;
        reader.reset(data);
        const sid = reader.vu57();
        const time = reader.vu57();
        const isServerClock = sid === 1;
        const clock = isServerClock ? new clock_1.ServerVectorClock(1, time) : new clock_1.VectorClock(sid, time);
        this.patchSid = clock.sid;
        const builder = (this.builder = new PatchBuilder_1.PatchBuilder(clock));
        const map = this.val();
        if (map instanceof Array)
            builder.patch.meta = map[0];
        this.decodeOperations();
        return builder.patch;
    }
    decodeId() {
        const reader = this.reader;
        const [isSessionDifferent, x] = reader.b1vu56();
        return isSessionDifferent ? new clock_1.Timestamp(reader.vu57(), x) : new clock_1.Timestamp(this.patchSid, x);
    }
    decodeTss() {
        const id = this.decodeId();
        const span = this.reader.vu57();
        return (0, clock_1.interval)(id, 0, span);
    }
    decodeOperations() {
        const reader = this.reader;
        const length = reader.vu57();
        for (let i = 0; i < length; i++)
            this.decodeOperation();
    }
    decodeOperation() {
        const builder = this.builder;
        const reader = this.reader;
        const octet = reader.u8();
        const opcode = octet & 0b11111;
        switch (opcode) {
            case 0: {
                const length = octet >> 5;
                builder.const(length === 0 ? this.val() : this.decodeId());
                break;
            }
            case 1: {
                builder.val();
                break;
            }
            case 2: {
                builder.obj();
                break;
            }
            case 3: {
                builder.vec();
                break;
            }
            case 4: {
                builder.str();
                break;
            }
            case 5: {
                builder.bin();
                break;
            }
            case 6: {
                builder.arr();
                break;
            }
            case 9: {
                const obj = this.decodeId();
                const val = this.decodeId();
                builder.setVal(obj, val);
                break;
            }
            case 10: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                const obj = this.decodeId();
                const tuples = [];
                for (let i = 0; i < length; i++) {
                    const key = this.val();
                    if (typeof key !== 'string')
                        continue;
                    const value = this.decodeId();
                    tuples.push([key, value]);
                }
                builder.insObj(obj, tuples);
                break;
            }
            case 11: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                const obj = this.decodeId();
                const tuples = [];
                for (let i = 0; i < length; i++) {
                    const index = this.val();
                    if (typeof index !== 'number')
                        continue;
                    const value = this.decodeId();
                    tuples.push([index, value]);
                }
                builder.insVec(obj, tuples);
                break;
            }
            case 12: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                const obj = this.decodeId();
                const after = this.decodeId();
                const str = reader.utf8(length);
                builder.insStr(obj, after, str);
                break;
            }
            case 13: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                const obj = this.decodeId();
                const after = this.decodeId();
                const buf = reader.buf(length);
                if (!(buf instanceof Uint8Array))
                    return;
                builder.insBin(obj, after, buf);
                break;
            }
            case 14: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                const obj = this.decodeId();
                const after = this.decodeId();
                const elements = [];
                for (let i = 0; i < length; i++)
                    elements.push(this.decodeId());
                builder.insArr(obj, after, elements);
                break;
            }
            case 16: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                const obj = this.decodeId();
                const what = [];
                for (let i = 0; i < length; i++)
                    what.push(this.decodeTss());
                builder.del(obj, what);
                break;
            }
            case 17: {
                let length = octet >> 5;
                if (length === 0)
                    length = reader.vu57();
                builder.nop(length);
                break;
            }
            default: {
                throw new Error('UNKNOWN_OP');
            }
        }
    }
}
exports.Decoder = Decoder;
