"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RpcPersistentClient = void 0;
const rxjs_1 = require("rxjs");
const operators_1 = require("rxjs/operators");
const StreamingRpcClient_1 = require("./client/StreamingRpcClient");
const ApiRpcCaller_1 = require("./caller/ApiRpcCaller");
const RpcDuplex_1 = require("../rpc/RpcDuplex");
const RpcMessageStreamProcessor_1 = require("./RpcMessageStreamProcessor");
const channel_1 = require("../channel");
class RpcPersistentClient {
    constructor(params) {
        this.rpc$ = new rxjs_1.ReplaySubject(1);
        const ping = params.ping ?? 15000;
        const codec = params.codec;
        const textEncoder = new TextEncoder();
        this.channel = new channel_1.PersistentChannel(params.channel);
        this.channel.open$.pipe((0, operators_1.filter)((open) => open)).subscribe(() => {
            const close$ = this.channel.open$.pipe((0, operators_1.filter)((open) => !open));
            const duplex = new RpcDuplex_1.RpcDuplex({
                client: new StreamingRpcClient_1.StreamingRpcClient({
                    ...(params.client || {}),
                    send: (messages) => {
                        const encoded = codec.encode(messages);
                        this.channel.send$(encoded).subscribe();
                    },
                }),
                server: new RpcMessageStreamProcessor_1.RpcMessageStreamProcessor({
                    ...(params.server || {
                        caller: new ApiRpcCaller_1.ApiRpcCaller({
                            api: {},
                        }),
                        onNotification: () => { },
                    }),
                    send: (messages) => {
                        const encoded = codec.encode(messages);
                        this.channel.send$(encoded).subscribe();
                    },
                }),
            });
            this.channel.message$.pipe((0, operators_1.takeUntil)(close$)).subscribe((data) => {
                const encoded = typeof data === 'string' ? textEncoder.encode(data) : new Uint8Array(data);
                const messages = codec.decode(encoded);
                duplex.onMessages((messages instanceof Array ? messages : [messages]), {});
            });
            if (ping) {
                (0, rxjs_1.timer)(ping, ping)
                    .pipe((0, operators_1.takeUntil)(close$))
                    .subscribe(() => {
                    duplex.notify('.ping', undefined);
                });
            }
            if (this.rpc)
                this.rpc.disconnect();
            this.rpc = duplex;
            this.rpc$.next(duplex);
        });
    }
    call$(method, data) {
        return this.rpc$.pipe((0, operators_1.first)(), (0, operators_1.switchMap)((rpc) => rpc.call$(method, data)), (0, operators_1.share)());
    }
    call(method, data) {
        return (0, rxjs_1.firstValueFrom)(this.call$(method, data));
    }
    notify(method, data) {
        this.rpc$.subscribe((rpc) => rpc.notify(method, data));
    }
    start() {
        this.channel.start();
    }
    stop() {
        this.channel.stop();
        if (this.rpc)
            this.rpc.stop();
    }
}
exports.RpcPersistentClient = RpcPersistentClient;
