"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.arithmeticOperators = void 0;
const tslib_1 = require("tslib");
const util = tslib_1.__importStar(require("../util"));
const codegen_steps_1 = require("../codegen-steps");
const toNum = util.num;
exports.arithmeticOperators = [
    [
        '+',
        ['add'],
        -1,
        (expr, ctx) => {
            return expr.slice(1).reduce((acc, e) => toNum(ctx.eval(e, ctx)) + acc, 0);
        },
        (ctx) => {
            const js = ctx.operands.map((expr) => `(+(${expr})||0)`).join('+');
            return new codegen_steps_1.Expression(js);
        },
    ],
    [
        '-',
        ['subtract'],
        -1,
        (expr, ctx) => {
            return expr.slice(2).reduce((acc, e) => acc - toNum(ctx.eval(e, ctx)), toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            const js = ctx.operands.map((expr) => `(+(${expr})||0)`).join('-');
            return new codegen_steps_1.Expression(js);
        },
    ],
    [
        '*',
        ['multiply'],
        -1,
        (expr, ctx) => {
            return expr.slice(1).reduce((acc, e) => toNum(ctx.eval(e, ctx)) * acc, 1);
        },
        (ctx) => {
            const js = ctx.operands.map((expr) => `(+(${expr})||0)`).join('*');
            return new codegen_steps_1.Expression(js);
        },
    ],
    [
        '/',
        ['divide'],
        -1,
        (expr, ctx) => {
            const start = toNum(ctx.eval(expr[1], ctx));
            return expr.slice(2).reduce((acc, e) => util.slash(acc, toNum(ctx.eval(e, ctx))), start);
        },
        (ctx) => {
            ctx.link(util.slash, 'slash');
            const params = ctx.operands.map((expr) => `(+(${expr})||0)`);
            let last = params[0];
            for (let i = 1; i < params.length; i++)
                last = `slash(${last}, ${params[i]})`;
            return new codegen_steps_1.Expression(last);
        },
    ],
    [
        '%',
        ['mod'],
        -1,
        (expr, ctx) => {
            const start = toNum(ctx.eval(expr[1], ctx));
            return expr.slice(2).reduce((acc, e) => util.mod(acc, toNum(ctx.eval(e, ctx))), start);
        },
        (ctx) => {
            ctx.link(util.mod, 'mod');
            const params = ctx.operands.map((expr) => `(+(${expr})||0)`);
            let last = params[0];
            for (let i = 1; i < params.length; i++)
                last = `mod(${last}, ${params[i]})`;
            return new codegen_steps_1.Expression(last);
        },
    ],
    [
        'min',
        [],
        -1,
        (expr, ctx) => {
            return Math.min(...expr.slice(1).map((e) => toNum(ctx.eval(e, ctx))));
        },
        (ctx) => {
            const params = ctx.operands.map((expr) => `(+(${expr})||0)`);
            return new codegen_steps_1.Expression(`+Math.min(${params.join(',')})||0`);
        },
    ],
    [
        'max',
        [],
        -1,
        (expr, ctx) => {
            return Math.max(...expr.slice(1).map((e) => toNum(ctx.eval(e, ctx))));
        },
        (ctx) => {
            const params = ctx.operands.map((expr) => `(+(${expr})||0)`);
            return new codegen_steps_1.Expression(`+Math.max(${params.join(',')})||0`);
        },
    ],
    [
        'round',
        [],
        1,
        (expr, ctx) => {
            return Math.round(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.round(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'ceil',
        [],
        1,
        (expr, ctx) => {
            return Math.ceil(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.ceil(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'floor',
        [],
        1,
        (expr, ctx) => {
            return Math.floor(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.floor(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'trunc',
        [],
        1,
        (expr, ctx) => {
            return Math.trunc(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.trunc(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'abs',
        [],
        1,
        (expr, ctx) => {
            return Math.abs(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.abs(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'sqrt',
        [],
        1,
        (expr, ctx) => {
            return Math.sqrt(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.sqrt(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'exp',
        [],
        1,
        (expr, ctx) => {
            return Math.exp(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.exp(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'ln',
        [],
        1,
        (expr, ctx) => {
            return Math.log(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.log(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        'log',
        [],
        2,
        (expr, ctx) => {
            const num = toNum(ctx.eval(expr[1], ctx));
            const base = toNum(ctx.eval(expr[2], ctx));
            return Math.log(num) / Math.log(base);
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.log(+(${ctx.operands[0]})||0)/Math.log(+(${ctx.operands[1]})||0)`);
        },
    ],
    [
        'log10',
        [],
        1,
        (expr, ctx) => {
            return Math.log10(toNum(ctx.eval(expr[1], ctx)));
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.log10(+(${ctx.operands[0]})||0)`);
        },
    ],
    [
        '**',
        ['pow'],
        2,
        (expr, ctx) => {
            const num = toNum(ctx.eval(expr[1], ctx));
            const base = toNum(ctx.eval(expr[2], ctx));
            return Math.pow(num, base);
        },
        (ctx) => {
            return new codegen_steps_1.Expression(`Math.pow(+(${ctx.operands[0]})||0,+(${ctx.operands[0]})||0)`);
        },
    ],
];
