"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ArrApi = exports.BinApi = exports.StrApi = exports.ObjApi = exports.VecApi = exports.ValApi = exports.ConApi = exports.NodeApi = void 0;
const find_1 = require("./find");
const clock_1 = require("../../../json-crdt-patch/clock");
const nodes_1 = require("../../nodes");
const NodeEvents_1 = require("./events/NodeEvents");
const printTree_1 = require("../../../util/print/printTree");
class NodeApi {
    constructor(node, api) {
        this.node = node;
        this.api = api;
        this.ev = undefined;
    }
    get events() {
        const et = this.ev;
        return et || (this.ev = new NodeEvents_1.NodeEvents(this));
    }
    find(path) {
        const node = this.node;
        if (path === undefined) {
            if (typeof node.child === 'function') {
                const child = node.child();
                if (!child)
                    throw new Error('NO_CHILD');
                return child;
            }
            throw new Error('CANNOT_IN');
        }
        if (typeof path === 'string' && !!path && path[0] !== '/')
            path = '/' + path;
        if (typeof path === 'number')
            path = [path];
        return (0, find_1.find)(this.node, path);
    }
    in(path) {
        const node = this.find(path);
        return this.api.wrap(node);
    }
    asVal() {
        if (this.node instanceof nodes_1.ValNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_VAL');
    }
    asStr() {
        if (this.node instanceof nodes_1.StrNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_STR');
    }
    asBin() {
        if (this.node instanceof nodes_1.BinNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_BIN');
    }
    asArr() {
        if (this.node instanceof nodes_1.ArrNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_ARR');
    }
    asTup() {
        if (this.node instanceof nodes_1.VecNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_ARR');
    }
    asObj() {
        if (this.node instanceof nodes_1.ObjNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_OBJ');
    }
    asCon() {
        if (this.node instanceof nodes_1.ConNode)
            return this.api.wrap(this.node);
        throw new Error('NOT_CONST');
    }
    asExt(ext) {
        let node = this.node;
        while (node) {
            if (node instanceof ext.Node)
                return new ext.Api(node, this.api);
            node = node.child ? node.child() : undefined;
        }
        throw new Error('NOT_EXT');
    }
    val(path) {
        return this.in(path).asVal();
    }
    str(path) {
        return this.in(path).asStr();
    }
    bin(path) {
        return this.in(path).asBin();
    }
    arr(path) {
        return this.in(path).asArr();
    }
    tup(path) {
        return this.in(path).asTup();
    }
    obj(path) {
        return this.in(path).asObj();
    }
    const(path) {
        return this.in(path).asCon();
    }
    view() {
        return this.node.view();
    }
    toString(tab = '') {
        return this.constructor.name + (0, printTree_1.printTree)(tab, [(tab) => this.node.toString(tab)]);
    }
}
exports.NodeApi = NodeApi;
class ConApi extends NodeApi {
    proxy() {
        return {
            toApi: () => this,
        };
    }
}
exports.ConApi = ConApi;
class ValApi extends NodeApi {
    get() {
        return this.in();
    }
    set(json) {
        const { api, node } = this;
        const builder = api.builder;
        const val = builder.constOrJson(json);
        api.builder.setVal(node.id, val);
        api.apply();
        return this;
    }
    proxy() {
        const self = this;
        const proxy = {
            toApi: () => this,
            get val() {
                const childNode = self.node.node();
                return self.api.wrap(childNode).proxy();
            },
        };
        return proxy;
    }
}
exports.ValApi = ValApi;
class VecApi extends NodeApi {
    get(key) {
        return this.in(key);
    }
    set(entries) {
        const { api, node } = this;
        const { builder } = api;
        builder.insVec(node.id, entries.map(([index, json]) => [index, builder.constOrJson(json)]));
        api.apply();
        return this;
    }
    proxy() {
        const proxy = new Proxy({}, {
            get: (target, prop, receiver) => {
                if (prop === 'toApi')
                    return () => this;
                const index = Number(prop);
                if (Number.isNaN(index))
                    throw new Error('INVALID_INDEX');
                const child = this.node.get(index);
                if (!child)
                    throw new Error('OUT_OF_BOUNDS');
                return this.api.wrap(child).proxy();
            },
        });
        return proxy;
    }
}
exports.VecApi = VecApi;
class ObjApi extends NodeApi {
    get(key) {
        return this.in(key);
    }
    set(entries) {
        const { api, node } = this;
        const { builder } = api;
        builder.insObj(node.id, Object.entries(entries).map(([key, json]) => [key, builder.constOrJson(json)]));
        api.apply();
        return this;
    }
    del(keys) {
        const { api, node } = this;
        const { builder } = api;
        api.builder.insObj(node.id, keys.map((key) => [key, builder.const(undefined)]));
        api.apply();
        return this;
    }
    proxy() {
        const self = this;
        const proxy = new Proxy({}, {
            get: (target, prop, receiver) => {
                if (prop === 'toApi')
                    return () => self;
                const key = String(prop);
                const child = this.node.get(key);
                if (!child)
                    throw new Error('NO_SUCH_KEY');
                return this.api.wrap(child).proxy();
            },
        });
        return proxy;
    }
}
exports.ObjApi = ObjApi;
class StrApi extends NodeApi {
    ins(index, text) {
        const { api, node } = this;
        const builder = api.builder;
        builder.pad();
        const nextTime = api.builder.nextTime();
        const id = new clock_1.Timestamp(builder.clock.sid, nextTime);
        const after = node.insAt(index, id, text);
        if (!after)
            throw new Error('OUT_OF_BOUNDS');
        builder.insStr(node.id, after, text);
        api.advance();
        return this;
    }
    del(index, length) {
        const { api, node } = this;
        const builder = api.builder;
        builder.pad();
        const spans = node.findInterval(index, length);
        if (!spans)
            throw new Error('OUT_OF_BOUNDS');
        node.delete(spans);
        builder.del(node.id, spans);
        api.advance();
        return this;
    }
    proxy() {
        return {
            toApi: () => this,
        };
    }
}
exports.StrApi = StrApi;
class BinApi extends NodeApi {
    ins(index, data) {
        const { api, node } = this;
        const after = !index ? node.id : node.find(index - 1);
        if (!after)
            throw new Error('OUT_OF_BOUNDS');
        api.builder.insBin(node.id, after, data);
        api.apply();
        return this;
    }
    del(index, length) {
        const { api, node } = this;
        const spans = node.findInterval(index, length);
        if (!spans)
            throw new Error('OUT_OF_BOUNDS');
        api.builder.del(node.id, spans);
        api.apply();
        return this;
    }
    proxy() {
        return {
            toApi: () => this,
        };
    }
}
exports.BinApi = BinApi;
class ArrApi extends NodeApi {
    get(index) {
        return this.in(index);
    }
    ins(index, values) {
        const { api, node } = this;
        const { builder } = api;
        const after = !index ? node.id : node.find(index - 1);
        if (!after)
            throw new Error('OUT_OF_BOUNDS');
        const valueIds = [];
        for (let i = 0; i < values.length; i++)
            valueIds.push(builder.json(values[i]));
        builder.insArr(node.id, after, valueIds);
        api.apply();
        return this;
    }
    del(index, length) {
        const { api, node } = this;
        const spans = node.findInterval(index, length);
        if (!spans)
            throw new Error('OUT_OF_BOUNDS');
        api.builder.del(node.id, spans);
        api.apply();
        return this;
    }
    length() {
        return this.node.length();
    }
    proxy() {
        const proxy = new Proxy({}, {
            get: (target, prop, receiver) => {
                if (prop === 'toApi')
                    return () => this;
                const index = Number(prop);
                if (Number.isNaN(index))
                    throw new Error('INVALID_INDEX');
                const child = this.node.getNode(index);
                if (!child)
                    throw new Error('OUT_OF_BOUNDS');
                return this.api.wrap(child).proxy();
            },
        });
        return proxy;
    }
}
exports.ArrApi = ArrApi;
