"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModelApi = void 0;
const nodes_1 = require("../../nodes");
const nodes_2 = require("./nodes");
const Emitter_1 = require("../../../util/events/Emitter");
const PatchBuilder_1 = require("../../../json-crdt-patch/PatchBuilder");
class ModelApi {
    constructor(model) {
        this.model = model;
        this.next = 0;
        this.changeQueued = false;
        this.queueChange = () => {
            if (this.changeQueued)
                return;
            this.changeQueued = true;
            queueMicrotask(() => {
                this.changeQueued = false;
                const et = this.et;
                if (et)
                    et.emit(new CustomEvent('change'));
            });
        };
        this.et = undefined;
        this.builder = new PatchBuilder_1.PatchBuilder(this.model.clock);
    }
    get events() {
        let et = this.et;
        if (!et) {
            this.et = et = new Emitter_1.Emitter();
            this.model.onchange = this.queueChange;
        }
        return et;
    }
    wrap(node) {
        if (node instanceof nodes_1.ValNode)
            return node.api || (node.api = new nodes_2.ValApi(node, this));
        else if (node instanceof nodes_1.StrNode)
            return node.api || (node.api = new nodes_2.StrApi(node, this));
        else if (node instanceof nodes_1.BinNode)
            return node.api || (node.api = new nodes_2.BinApi(node, this));
        else if (node instanceof nodes_1.ArrNode)
            return node.api || (node.api = new nodes_2.ArrApi(node, this));
        else if (node instanceof nodes_1.ObjNode)
            return node.api || (node.api = new nodes_2.ObjApi(node, this));
        else if (node instanceof nodes_1.ConNode)
            return node.api || (node.api = new nodes_2.ConApi(node, this));
        else if (node instanceof nodes_1.VecNode)
            return node.api || (node.api = new nodes_2.VecApi(node, this));
        else
            throw new Error('UNKNOWN_NODE');
    }
    get r() {
        return new nodes_2.ValApi(this.model.root, this);
    }
    get node() {
        return this.r.get();
    }
    in(path) {
        return this.r.in(path);
    }
    find(path) {
        return this.node.find(path);
    }
    val(path) {
        return this.node.val(path);
    }
    vec(path) {
        return this.node.tup(path);
    }
    str(path) {
        return this.node.str(path);
    }
    bin(path) {
        return this.node.bin(path);
    }
    arr(path) {
        return this.node.arr(path);
    }
    obj(path) {
        return this.node.obj(path);
    }
    const(path) {
        return this.node.const(path);
    }
    root(json) {
        const builder = this.builder;
        builder.root(builder.json(json));
        this.apply();
        return this;
    }
    apply() {
        const ops = this.builder.patch.ops;
        const length = ops.length;
        const model = this.model;
        for (let i = this.next; i < length; i++)
            model.applyOperation(ops[i]);
        this.next = length;
        model.tick++;
        model.onchange?.();
    }
    advance() {
        this.next = this.builder.patch.ops.length;
        const model = this.model;
        model.tick++;
        model.onchange?.();
    }
    view() {
        return this.model.view();
    }
    flush() {
        const patch = this.builder.flush();
        this.next = 0;
        return patch;
    }
}
exports.ModelApi = ModelApi;
