"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Decoder = void 0;
const tslib_1 = require("tslib");
const nodes = tslib_1.__importStar(require("../../../nodes"));
const ClockTable_1 = require("../../../../json-crdt-patch/codec/clock/ClockTable");
const CrdtReader_1 = require("../../../../json-crdt-patch/util/binary/CrdtReader");
const clock_1 = require("../../../../json-crdt-patch/clock");
const Model_1 = require("../../../model/Model");
const CborDecoderBase_1 = require("../../../../json-pack/cbor/CborDecoderBase");
class Decoder {
    constructor(reader) {
        this.decodeStrChunk = () => {
            const decoder = this.dec;
            const reader = decoder.reader;
            const id = this.ts();
            const isTombstone = reader.uint8[reader.x] === 0;
            if (isTombstone) {
                reader.x++;
                const length = reader.vu39();
                return new nodes.StrChunk(id, length, '');
            }
            const text = decoder.readAsStr();
            return new nodes.StrChunk(id, text.length, text);
        };
        this.decodeBinChunk = () => {
            const id = this.ts();
            const reader = this.dec.reader;
            const [deleted, length] = reader.b1vu56();
            if (deleted)
                return new nodes.BinChunk(id, length, undefined);
            else
                return new nodes.BinChunk(id, length, reader.buf(length));
        };
        this.decodeArrChunk = () => {
            const id = this.ts();
            const reader = this.dec.reader;
            const [deleted, length] = reader.b1vu56();
            if (deleted)
                return new nodes.ArrChunk(id, length, undefined);
            else {
                const data = [];
                for (let i = 0; i < length; i++)
                    data.push(this.ts());
                return new nodes.ArrChunk(id, length, data);
            }
        };
        this.dec = new CborDecoderBase_1.CborDecoderBase(reader || new CrdtReader_1.CrdtReader());
    }
    decode(fields, ModelConstructor = Model_1.Model) {
        const reader = this.dec.reader;
        reader.reset(fields.c);
        const clockTable = (this.clockTable = ClockTable_1.ClockTable.decode(reader));
        return this.decodeFields(clockTable, fields, ModelConstructor);
    }
    decodeFields(clockTable, fields, ModelConstructor = Model_1.Model) {
        const reader = this.dec.reader;
        const firstClock = clockTable.byIdx[0];
        const vectorClock = new clock_1.VectorClock(firstClock.sid, firstClock.time + 1);
        const doc = (this.doc = new ModelConstructor(vectorClock));
        const root = fields.r;
        if (root && root.length) {
            reader.reset(root);
            const rootValue = this.ts();
            doc.root.set(rootValue);
        }
        const index = doc.index;
        const keys = Object.keys(fields);
        const length = keys.length;
        for (let i = 0; i < length; i++) {
            const field = keys[i];
            if (field.length < 3)
                continue;
            const arr = fields[field];
            const id = clockTable.parseField(field);
            reader.reset(arr);
            const node = this.decodeNode(id);
            index.set(id, node);
        }
        return doc;
    }
    ts() {
        const [sessionIndex, timeDiff] = this.dec.reader.id();
        return new clock_1.Timestamp(this.clockTable.byIdx[sessionIndex].sid, timeDiff);
    }
    decodeNode(id) {
        const reader = this.dec.reader;
        const octet = reader.u8();
        const major = octet >> 5;
        const minor = octet & 0b11111;
        const length = minor < 24 ? minor : minor === 24 ? reader.u8() : minor === 25 ? reader.u16() : reader.u32();
        switch (major) {
            case 0:
                return this.decodeCon(id, length);
            case 1:
                return this.decodeVal(id);
            case 2:
                return this.decodeObj(id, length);
            case 3:
                return this.decodeVec(id, length);
            case 4:
                return this.decodeStr(id, length);
            case 5:
                return this.decodeBin(id, length);
            case 6:
                return this.decodeArr(id, length);
        }
        return Model_1.UNDEFINED;
    }
    decodeCon(id, length) {
        const decoder = this.dec;
        const data = !length ? decoder.val() : this.ts();
        const node = new nodes.ConNode(id, data);
        return node;
    }
    decodeVal(id) {
        const val = this.ts();
        const node = new nodes.ValNode(this.doc, id, val);
        return node;
    }
    decodeObj(id, length) {
        const decoder = this.dec;
        const obj = new nodes.ObjNode(this.doc, id);
        const keys = obj.keys;
        for (let i = 0; i < length; i++) {
            const key = decoder.val() + '';
            const val = this.ts();
            keys.set(key, val);
        }
        return obj;
    }
    decodeVec(id, length) {
        const reader = this.dec.reader;
        const node = new nodes.VecNode(this.doc, id);
        const elements = node.elements;
        for (let i = 0; i < length; i++) {
            const octet = reader.u8();
            if (!octet)
                elements.push(undefined);
            else
                elements.push(this.ts());
        }
        return node;
    }
    decodeStr(id, length) {
        const node = new nodes.StrNode(id);
        node.ingest(length, this.decodeStrChunk);
        return node;
    }
    decodeBin(id, length) {
        const node = new nodes.BinNode(id);
        node.ingest(length, this.decodeBinChunk);
        return node;
    }
    decodeArr(id, length) {
        const node = new nodes.ArrNode(this.doc, id);
        node.ingest(length, this.decodeArrChunk);
        return node;
    }
}
exports.Decoder = Decoder;
