"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.defineCrdtRoutes = void 0;
const json_crdt_1 = require("../../json-crdt");
const Encoder_1 = require("../../json-crdt/codec/structural/verbose/Encoder");
const Encoder_2 = require("../../json-crdt/codec/structural/compact/Encoder");
const encode_1 = require("../../json-crdt-patch/codec/compact/encode");
const defineCrdtRoutes = (router) => {
    return router.extend(({ t }) => {
        return {
            'crdt.create': t
                .Function(t.Object(t.propOpt('value', t.any).options({
                title: 'Initial value',
                description: 'Initial value of the JSON CRDT document. Can be any JSON value.',
            }), t.propOpt('sid', t.num).options({
                title: 'Session ID',
                description: 'Session ID to use for the logic clock of the document. If not provided, a random number will be used.',
            }), t.propOpt('serverClock', t.bool).options({
                title: 'Use server clock',
                description: 'Whether to use server logical clock for this document. If set to true, the session ID will be ignored set to 1.',
            }), t
                .propOpt('codec', t.Or(t.Const('binary'), t.Const('compact'), t.Const('verbose')).options({
                discriminator: ['?', ['==', ['$', ''], 'binary'], 0, ['?', ['==', ['$', ''], 'compact'], 1, 2]],
            }))
                .options({
                title: 'Codec for the document',
                description: 'Codec to use for the document. Defaults to binary.',
            })), t.Object(t.prop('doc', t.any).options({
                title: 'JSON CRDT document',
                description: 'JSON CRDT document.',
            }), t.prop('codec', t.str), t.prop('patch', t.any)))
                .options({
                title: 'Create a CRDT document',
                description: 'Creates a new JSON CRDT document.',
            })
                .implement(async ({ value, sid, serverClock, codec }) => {
                const model = serverClock
                    ? json_crdt_1.Model.withServerClock()
                    : sid !== undefined
                        ? json_crdt_1.Model.withLogicalClock(sid)
                        : json_crdt_1.Model.withLogicalClock();
                if (value !== undefined)
                    model.api.root(value);
                const patch = model.api.flush();
                const patchEncoded = patch && patch.ops.length ? (0, encode_1.encode)(patch) : null;
                codec ?? (codec = 'binary');
                let doc = null;
                switch (codec) {
                    case 'binary': {
                        doc = model.toBinary();
                        break;
                    }
                    case 'compact': {
                        const encoder = new Encoder_2.Encoder();
                        doc = encoder.encode(model);
                        break;
                    }
                    case 'verbose': {
                        const encoder = new Encoder_1.Encoder();
                        doc = encoder.encode(model);
                        break;
                    }
                    default:
                        throw new Error(`Unknown codec: ${codec}`);
                }
                return {
                    doc,
                    codec,
                    patch: patchEncoded,
                };
            }),
        };
    });
};
exports.defineCrdtRoutes = defineCrdtRoutes;
