"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPABuilder = void 0;
const webpack_1 = require("webpack");
const builder_base_1 = require("./builder-base");
const path_1 = __importDefault(require("path"));
const html_webpack_plugin_1 = __importDefault(require("html-webpack-plugin"));
const mini_css_extract_plugin_1 = __importDefault(require("mini-css-extract-plugin"));
const terser_webpack_plugin_1 = __importDefault(require("terser-webpack-plugin"));
const case_1 = __importDefault(require("case"));
const fs_1 = __importDefault(require("fs"));
const webpack_virtual_modules_1 = __importDefault(require("webpack-virtual-modules"));
const generate_auto_loader_file_1 = require("./utils/generate-auto-loader-file");
const react_refresh_webpack_plugin_1 = __importDefault(require("@pmmmwh/react-refresh-webpack-plugin"));
/**
 * SPA Builder
 */
class SPABuilder extends builder_base_1.BuilderBase {
    /**
     * Creates a new SPA builder instance
     * @param {string} id
     * @param {string} appFilePath
     */
    constructor(id, appFilePath) {
        super();
        this.appId = id;
        this.appFilePath = appFilePath;
        this.virtualModules = new webpack_virtual_modules_1.default({
            [path_1.default.join('src', `${this.appId}.tsx`)]: `
        import { startApp } from '@magicjs.dev/frontend';
        import { initializeModules } from './auto-loader.tsx';

        initializeModules();
        startApp();
      `
        });
        if (!this.appId) {
            throw new Error('App ID should not be null');
        }
    }
    /**
     * Get Ghost Files
     * @param {ConfigurationOptions} opts
     * @return {GhostFileActions[]}
     */
    getGhostFiles(opts) {
        let importExpressions = '';
        let registrationExpressions = '';
        const arkJSONPath = path_1.default.join(opts.cwd, 'src', 'app.json');
        const arkJSON = JSON.parse(fs_1.default.readFileSync(arkJSONPath, 'utf-8'));
        if (Array.isArray(arkJSON.routes)) {
            const importables = arkJSON.routes.map((route) => {
                return {
                    filePath: `.${path_1.default.sep}${path_1.default.relative(path_1.default.join(opts.cwd, 'src'), path_1.default.join(opts.cwd, 'src', route.view))}`,
                    fileId: case_1.default.camel(route.view)
                };
            });
            importExpressions = importables.map((importable) => `import ${importable.fileId} from '${importable.filePath}';`).join('\n');
            registrationExpressions = importables.map((importable) => `registerView('${importable.fileId}', ${importable.fileId});`).join('\n');
        }
        return [];
    }
    initCompiler(opts) {
        this.compiler.hooks.compilation.tap('MyPlugin', (compilation) => {
            const content = (0, generate_auto_loader_file_1.generateAutoloaderFile)(opts.cwd, 'frontend');
            this.virtualModules.writeModule(path_1.default.join('src', 'auto-loader.tsx'), content);
        });
    }
    /**
     * @param {ConfigurationOptions} opts
     * @return {Configuration}
     */
    getConfiguration({ cwd, mode }) {
        const babelLoaderOptions = {
            // Should not take any babelrc file located in the project root
            babelrc: false,
            sourceMaps: mode === 'development' ? 'inline' : false,
            compact: mode === 'production',
            presets: [
                [
                    require.resolve('@babel/preset-env'),
                    {
                        targets: { browsers: ['last 2 versions'] },
                        modules: false,
                    },
                ],
                [
                    require.resolve('@babel/preset-typescript'),
                    { allowNamespaces: true },
                ],
                [require.resolve('@babel/preset-react')],
            ],
            cacheDirectory: true,
            cacheCompression: false,
            plugins: [
                require.resolve('@babel/plugin-proposal-class-properties'),
                require.resolve('@babel/plugin-syntax-dynamic-import'),
                mode === 'development' && require.resolve('react-refresh/babel')
            ].filter(Boolean),
        };
        return {
            cache: true,
            devtool: mode === 'development' ? 'source-map' : false,
            context: cwd,
            mode,
            resolve: {
                fallback: {
                    'path': require.resolve('path-browserify')
                },
                modules: ['scripts', 'node_modules'],
                extensions: ['.json', '.ts', '.tsx', '.js', '.jsx'],
                alias: Object.assign({}, this.mapPeerDependencies(['react', 'react-dom', 'react-router-dom'], cwd)),
                symlinks: true,
            },
            entry: {
                [this.appId]: [
                    mode === 'development' && require.resolve('webpack-hot-middleware/client'),
                    path_1.default.join(cwd, 'src', `client.tsx`)
                ].filter(Boolean),
            },
            output: {
                publicPath: '/',
                filename: `_browser/[name].js`,
                path: path_1.default.resolve(cwd, 'build'),
                assetModuleFilename: './assets/[hash][ext][query]',
                chunkFilename: '_browser/[name].[contenthash].js',
            },
            plugins: [
                new html_webpack_plugin_1.default({
                    filename: '[name].html',
                    template: path_1.default.resolve(__dirname, '../assets/index.template.html'),
                }),
                new mini_css_extract_plugin_1.default({
                    filename: './assets/[name].css',
                    chunkFilename: './assets/[name].[contenthash:8].chunk.css',
                }),
                this.virtualModules,
                mode === 'development' && new webpack_1.HotModuleReplacementPlugin({
                    autoConnect: true,
                    timeout: 4000
                }),
                mode === 'development' && new react_refresh_webpack_plugin_1.default({
                    overlay: false,
                    forceEnable: true
                })
            ].filter(Boolean),
            stats: {
                loggingTrace: false,
                errorStack: false,
            },
            resolveLoader: {
                modules: ['node_modules', path_1.default.resolve(__dirname, 'loaders')]
            },
            module: {
                rules: [
                    {
                        test: /\.js$/,
                        enforce: 'pre',
                        use: [require.resolve('source-map-loader')],
                    },
                    {
                        test: /\.(js|mjs|jsx|ts|tsx)$/,
                        exclude: /node_modules/,
                        use: [
                            {
                                loader: require.resolve('babel-loader'),
                                options: babelLoaderOptions,
                            },
                            'remote-loader'
                        ],
                    },
                    {
                        test: /\.svg(\?v=\d+\.\d+\.\d+)?$/,
                        use: [
                            {
                                loader: require.resolve('babel-loader'),
                                options: babelLoaderOptions,
                            },
                            {
                                loader: require.resolve('@svgr/webpack'),
                                options: {
                                    babel: false,
                                    icon: true,
                                },
                            },
                        ],
                    },
                    {
                        test: /\.(png|jpg|jpeg|gif|woff2|woff|ttf)$/i,
                        use: [
                            {
                                loader: require.resolve('file-loader'),
                                options: {
                                    name: '[contenthash].[ext]',
                                    outputPath: 'assets',
                                },
                            },
                        ],
                    },
                    {
                        test: this.getStyleTestExp(),
                        use: [
                            (() => {
                                if (mode === 'development') {
                                    return {
                                        loader: require.resolve('style-loader')
                                    };
                                }
                                return {
                                    loader: mini_css_extract_plugin_1.default.loader
                                };
                            })(),
                            {
                                loader: require.resolve('css-loader'),
                            },
                            {
                                loader: require.resolve('postcss-loader'),
                            },
                            {
                                loader: require.resolve('sass-loader'),
                            }
                        ],
                    },
                ],
            },
            performance: {
                maxEntrypointSize: 5242880,
                maxAssetSize: 5242880,
            },
            optimization: {
                minimize: mode === 'production',
                minimizer: [new terser_webpack_plugin_1.default()],
                splitChunks: {
                    chunks: 'async',
                    minSize: 20000,
                    minRemainingSize: 0,
                    maxSize: 5242880,
                    minChunks: 1,
                    maxAsyncRequests: 30,
                    maxInitialRequests: 30,
                    enforceSizeThreshold: 5242880,
                    cacheGroups: {
                        defaultVendors: {
                            test: /[\\/]node_modules[\\/]/,
                            priority: -10,
                            reuseExistingChunk: true,
                        },
                        default: {
                            minChunks: 2,
                            priority: -20,
                            reuseExistingChunk: true,
                        },
                    },
                },
                runtimeChunk: {
                    name: (entrypoint) => `runtime-${entrypoint.name}`,
                },
            },
        };
    }
}
exports.SPABuilder = SPABuilder;
