"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.runBuild = void 0;
const express_1 = __importDefault(require("express"));
const path_1 = __importDefault(require("path"));
const frontend_builder_1 = require("./frontend-builder");
const backend_builder_1 = require("./backend-builder");
const webpack_dev_middleware_1 = __importDefault(require("webpack-dev-middleware"));
const webpack_hot_middleware_1 = __importDefault(require("webpack-hot-middleware"));
const child_process_1 = require("child_process");
const fs_extra_1 = __importDefault(require("fs-extra"));
const http_proxy_1 = __importDefault(require("http-proxy"));
const chalk_1 = __importDefault(require("chalk"));
const get_port_1 = __importDefault(require("get-port"));
const open_1 = __importDefault(require("open"));
const axios_1 = __importDefault(require("axios"));
const json_stringify_safe_1 = __importDefault(require("json-stringify-safe"));
const os_1 = __importDefault(require("os"));
const proxy = http_proxy_1.default.createProxyServer();
proxy.on('error', (err, req, res) => {
    console.log('Could not connect to server process');
});
function runBuild(opts) {
    return __awaiter(this, void 0, void 0, function* () {
        let { runtimeUrl, env } = opts;
        if (env === undefined) {
            env = 'development';
        }
        // Clean up build folder
        fs_extra_1.default.emptyDirSync(path_1.default.join(opts.cwd, 'build'));
        const HasRuntimeAgent = Boolean(runtimeUrl);
        let status = {
            appServerPort: yield (0, get_port_1.default)(),
            appServerLive: false,
            devServerPort: 3000,
            devServerActive: false,
            frontendCompiled: false,
            backendCompiled: false,
            compilationStatus: 'building',
            frontendWarnings: [],
            backendWarnings: [],
            frontendErrors: [],
            backendErrors: [],
            hasWarnings: false,
            hasErrors: false,
            __frontendStatusInStr: '',
            __backendStatusInStr: '',
        };
        if (!isNaN(Number(process.env['PORT']))) {
            status.devServerPort = Number(process.env['PORT']);
        }
        let appProcess;
        let appRunningTimer = null;
        const runApp = () => {
            if (env !== 'development') {
                return;
            }
            status.appServerLive = false;
            clearTimeout(appRunningTimer);
            if (appProcess) {
                if (os_1.default.platform() === 'win32') {
                    appProcess.kill('SIGTERM');
                }
                else {
                    appProcess.kill('SIGKILL');
                }
            }
            const appPath = path_1.default.join(opts.cwd, 'build', 'server', 'main.js');
            if (!fs_extra_1.default.existsSync(appPath)) {
                console.log('');
                console.log('Waiting for output...');
                return false;
            }
            appProcess = (0, child_process_1.spawn)('node', [appPath], {
                // stdio: 'inherit',
                env: Object.assign(Object.assign({}, process.env), { DEV_PORT: String(status.appServerPort) })
            });
            const processLog = (msg) => {
                if (status.appServerLive === false) {
                    status.appServerLive = msg.startsWith('Listening on port');
                }
                console.log(msg);
            };
            appProcess.stdout.on('data', (data) => {
                let d = String(data);
                processLog(d.substring(0, d.length - 1));
            });
            appProcess.stderr.on('data', (data) => {
                let d = String(data);
                processLog(d.substring(0, d.length - 1));
            });
            appRunningTimer = setTimeout(() => {
                status.appServerLive = true;
            }, 3000);
        };
        const refreshCompilationStatus = () => {
            status.hasErrors = status.backendErrors.length > 0 || status.frontendErrors.length > 0;
            status.hasWarnings = status.backendWarnings.length > 0 || status.frontendWarnings.length > 0;
            if (status.backendCompiled === true && status.frontendCompiled === true) {
                if (status.hasErrors === true) {
                    status.compilationStatus = 'error';
                }
                else if (status.hasWarnings === true) {
                    status.compilationStatus = 'compiled-with-warnings';
                }
                else {
                    status.compilationStatus = 'ready';
                }
            }
            else {
                status.compilationStatus = 'building';
            }
            printLog();
            reportToRuntime(status);
        };
        const reportToRuntime = (payload) => {
            if (HasRuntimeAgent === true) {
                (0, axios_1.default)({
                    method: 'post',
                    baseURL: runtimeUrl,
                    url: '/__ark__agent_c/report_from_compiler',
                    data: (0, json_stringify_safe_1.default)(payload),
                    headers: {
                        'Content-Type': 'application/json',
                    }
                })
                    .catch((e) => {
                    console.log(chalk_1.default.red(`Could not communicate with agent ${runtimeUrl}`));
                });
            }
        };
        const printLog = () => {
            if (env !== 'development') {
                return;
            }
            console.clear();
            // Show compilation status
            switch (status.compilationStatus) {
                case 'ready': {
                    console.log(chalk_1.default.green('Compiled successfully!'));
                    break;
                }
                case 'error': {
                    console.log(chalk_1.default.red('Compilation failed'));
                    break;
                }
                case 'compiled-with-warnings': {
                    console.log(chalk_1.default.yellow('Compiled with warnings'));
                    console.log(status.backendWarnings);
                    console.log(status.frontendWarnings);
                    break;
                }
                default: {
                    console.log(chalk_1.default.blue('Building changes...'));
                    break;
                }
            }
            if (status.compilationStatus === 'ready' || status.compilationStatus === 'compiled-with-warnings') {
                console.log('');
                if (status.devServerActive === true) {
                    console.log('You can now view project in the browser');
                    console.log('');
                    console.log(`    Local:       http://localhost:${status.devServerPort}`);
                    console.log('');
                }
            }
            else {
                if (status.devServerActive === false && env === 'development') {
                    console.log('');
                    console.log(chalk_1.default.yellow(`Starting development server...`));
                }
            }
        };
        const backendBuilder = new backend_builder_1.BackendBuilder(path_1.default.join(opts.cwd, 'src/server.tsx'));
        const frontendBuilder = new frontend_builder_1.SPABuilder('client', path_1.default.join(opts.cwd, 'src/client.tsx'));
        backendBuilder.on('compiling', () => {
            status.backendCompiled = false;
            refreshCompilationStatus();
        });
        frontendBuilder.on('compiling', () => {
            status.frontendCompiled = false;
            refreshCompilationStatus();
        });
        backendBuilder.attachMonitor((err, result) => {
            try {
                if (err)
                    throw err;
                if (env === 'production') {
                    status.__backendStatusInStr = result.toString();
                }
                if (result) {
                    status.backendErrors = result.compilation.errors;
                    status.backendWarnings = result.compilation.warnings;
                }
                status.backendCompiled = true;
                refreshCompilationStatus();
                runApp();
            }
            catch (e) {
                console.error(e);
            }
        });
        frontendBuilder.attachMonitor((err, result) => {
            try {
                if (err)
                    throw err;
                if (env === 'production') {
                    status.__frontendStatusInStr = result.toString();
                }
                if (result) {
                    status.frontendErrors = result.compilation.errors;
                    status.frontendWarnings = result.compilation.warnings;
                }
                status.frontendCompiled = true;
                refreshCompilationStatus();
                runApp();
            }
            catch (e) {
                console.error(e);
            }
        });
        backendBuilder.build({
            cwd: opts.cwd,
            mode: env,
            watchMode: env === 'development'
        });
        frontendBuilder.build({
            cwd: opts.cwd,
            mode: env,
            watchMode: env === 'development'
        });
        if (env === 'development') {
            const app = (0, express_1.default)();
            printLog();
            const backendMiddleware = (0, webpack_dev_middleware_1.default)(backendBuilder.compiler, { stats: 'none', outputFileSystem: require('fs') });
            const frontendMiddleware = (0, webpack_dev_middleware_1.default)(frontendBuilder.compiler, { stats: 'none', outputFileSystem: require('fs') });
            app.use(backendMiddleware);
            app.use(frontendMiddleware);
            app.use((0, webpack_hot_middleware_1.default)(frontendBuilder.compiler, { log: false, heartbeat: 2000 }));
            app.get('/____compiler__status', (req, res) => {
                res.json(JSON.parse((0, json_stringify_safe_1.default)(status)));
            });
            app.all('/*', (req, res) => __awaiter(this, void 0, void 0, function* () {
                try {
                    while (status.appServerLive === false && status.hasErrors === false) {
                        yield new Promise((r) => setTimeout(() => r(), 300));
                    }
                    if (status.hasErrors === true) {
                        let backendOutput = null;
                        let frontendOutput = null;
                        backendOutput = [
                            ...backendMiddleware.context.stats.toJson().errors,
                            ...backendMiddleware.context.stats.toJson().warnings
                        ];
                        frontendOutput = [
                            ...frontendMiddleware.context.stats.toJson().errors,
                            ...frontendMiddleware.context.stats.toJson().warnings
                        ];
                        res.send(`
                    <html>
                    <head>
                    </head>
                    <body style="padding: 24px; background-color: white;">
                        <div style="background-color: white;">
                        <h1 style="margin: 0px; font-family: sans-serif; font-weight: normal;">There's some <span style="background-color: red; color: white; padding: 0 12px;">error</span> in the code</h1>
                        <h2 style="margin-top: 8px; font-family: sans-serif; font-weight: normal;">Please use the terminal or below output to further troubleshoot it</h2>

                        <button style="margin-top: 12px" onclick="reload()">Refresh</button>
                        <code style="margin-top: 12px">
                        ${(0, json_stringify_safe_1.default)(backendOutput)}
                        </code>
                        <code style="margin-top: 12px">
                        ${(0, json_stringify_safe_1.default)(frontendOutput)}
                        </code>
                        </div>
                        <script>
                            function reload() {
                                window.location.reload();
                            }
                        </script>
                    </body>
                    </html>
                `);
                        return;
                    }
                    res.setHeader("Cache-Control", "no-cache, no-store, must-revalidate"); // HTTP 1.1.
                    res.setHeader("Pragma", "no-cache"); // HTTP 1.0.
                    res.setHeader("Expires", "0"); // Proxies.
                    proxy.web(req, res, {
                        target: `http://localhost:${status.appServerPort}`,
                        ws: true
                    });
                }
                catch (e) {
                    res.status(500).json({ message: e === null || e === void 0 ? void 0 : e.message });
                }
            }));
            const server = app.listen(status.devServerPort, undefined, undefined, () => {
                status.devServerActive = true;
                printLog();
                if (HasRuntimeAgent === false) {
                    (0, open_1.default)(`http://localhost:${status.devServerPort}`);
                }
            });
            server.on('upgrade', (req, socket, head) => {
                proxy.ws(req, socket, head, {
                    target: `http://localhost:${status.appServerPort}`
                });
            });
        }
        else {
            let timer = setInterval(() => {
                if (status.frontendCompiled === true && status.backendCompiled === true) {
                    if (status.hasErrors === true) {
                        console.log(status.backendErrors);
                        console.log(status.frontendErrors);
                        console.error(chalk_1.default.red('Compiled with error'));
                        process.exit(1);
                    }
                    else {
                        console.log(status.__backendStatusInStr);
                        console.log(status.__frontendStatusInStr);
                        console.error(chalk_1.default.green('Built successfully'));
                        process.exit(0);
                    }
                }
            }, 1000);
        }
    });
}
exports.runBuild = runBuild;
