"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuilderBase = void 0;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const ejs_1 = __importDefault(require("ejs"));
const webpack_1 = __importDefault(require("webpack"));
const events_1 = require("events");
const memfs_1 = __importDefault(require("memfs"));
const unionfs_1 = require("unionfs");
/**
 * Wrapper for Webpack
 */
class BuilderBase extends events_1.EventEmitter {
    /**
     * Creates a new builder base instance
     * @param {EventEmitterOptions} options
     */
    constructor(options) {
        super(options);
    }
    initCompiler(opts) { }
    /**
     * Start build process
     * @param {ConfigurationOptions} opts
     * @param {any=} ifs Input filesystem
     * @param {any=} ofs Output filesystem
     * @param {any=} wfs Watch filesystem
     */
    build(opts, ifs, ofs, wfs) {
        this.options = Object.assign({
            mode: 'production',
            cwd: null,
            watchMode: false,
        }, opts);
        // Normalise cwd to support win32
        this.options.cwd = this.options.cwd.split(path_1.default.sep).join(path_1.default.posix.sep);
        const buildConfiguration = this.getConfiguration(this.options);
        if (!buildConfiguration) {
            throw new Error('webpack configuration should not be null');
        }
        this.compiler = (0, webpack_1.default)(buildConfiguration);
        if (ifs) {
            this.compiler.inputFileSystem = ifs;
        }
        if (ofs) {
            this.compiler.outputFileSystem = ofs;
        }
        if (wfs) {
            this.compiler.watchFileSystem = wfs;
        }
        // Set ghost files
        const volume = this.getGhostFiles(opts).reduce((acc, ghostFile) => {
            return Object.assign(Object.assign({}, acc), ghostFile.provide(opts.cwd));
        }, {});
        let _memfs;
        let _ufs;
        if (Object.keys(volume).length > 0) {
            _memfs = memfs_1.default.createFsFromVolume(memfs_1.default.Volume.fromJSON(volume, opts.cwd));
            _ufs = unionfs_1.ufs
                .use(_memfs)
                .use(fs_1.default);
            this.compiler.inputFileSystem = _ufs;
        }
        this.compiler.hooks.invalid.tap('invalid', () => {
            this.emit('compiling');
        });
        this.initCompiler(opts);
        if (opts.watchMode === true) {
            this.watching = this.compiler.watch({}, this.handler.bind(this));
        }
        else {
            this.compiler.run(this.handler.bind(this));
        }
    }
    /**
     * Teardown logic
     * @return {Promise}
     */
    teardown() {
        return new Promise((resolve, reject) => {
            this.removeAllListeners('success');
            this.removeAllListeners('warning');
            this.removeAllListeners('error');
            if (this.watching) {
                this.watching.close((err, result) => {
                    if (err) {
                        reject(err);
                    }
                    else {
                        resolve(true);
                    }
                });
            }
            else {
                resolve(true);
            }
        });
    }
    /**
     * Attaches a monitor that can be used to listen to events
     * @param {BuilderMonitor} mon
     */
    attachMonitor(mon) {
        this.monitor = mon;
    }
    /**
     * Gets input ghost files
     * @param {ConfigurationOptions} opts
     * @return {GhostFileActions[]}
     */
    getGhostFiles(opts) {
        return [];
    }
    /**
     * Supported events
     * @return {string[]}
     */
    eventNames() {
        return ['success', 'warning', 'error'];
    }
    /**
     * Gets configuration
     * @param {ConfigurationOptions} opts
     * @return {Configuration}
     */
    getConfiguration(opts) {
        return null;
    }
    /**
     * Gets stylesheet test expression
     * @return {RegExp}
     */
    getStyleTestExp() {
        return /\.(scss|css|sass)$/i;
    }
    /**
     * Gets LESS stylesheet test expression
     * @return {RegExp}
     */
    getLESSStyleTestExp() {
        return /\.less$/i;
    }
    /**
     * Create alias mapping with peer dependencies
     * @param {string[]} dependencies
     * @param {string=} cwd Defaults to process.cwd()
     * @return {any}
     */
    mapPeerDependencies(dependencies, cwd) {
        return dependencies.reduce((acc, dependency) => {
            cwd = cwd || process.cwd();
            let peerNodeModulesPath = path_1.default.resolve(cwd, 'node_modules', dependency);
            if (!fs_1.default.existsSync(peerNodeModulesPath)) {
                cwd = process.cwd();
                peerNodeModulesPath = path_1.default.resolve(cwd, 'node_modules', dependency);
            }
            return Object.assign({ [dependency]: peerNodeModulesPath }, acc);
        }, {});
    }
    /**
     * Generate file from template / retreives optional file
     * @param {string} cwd Current Working Directory
     * @param {string} relativePath Relative path of the file from project root
     * @param {string} ejsFilePath Template file path
     * @param {object=} data (Optional) template render options
     * @return {string} Optional file from project dir / template output
     */
    getOptionalFile(cwd, relativePath, ejsFilePath, data) {
        const optionalFile = path_1.default.join(cwd, relativePath);
        if (fs_1.default.existsSync(optionalFile)) {
            // Output read file from projects dir
            return fs_1.default.readFileSync(optionalFile, 'utf-8');
        }
        else {
            if (fs_1.default.existsSync(ejsFilePath)) {
                // Read template file
                const template = fs_1.default.readFileSync(ejsFilePath, 'utf-8');
                return ejs_1.default.render(template, data);
            }
            // eslint-disable-next-line max-len
            throw new Error('Failed to compile replacement file. This indicates an error with Ark Build System, you may create an issue for this on GitHub.');
        }
    }
    /**
     * Invokes monitor if one's attached
     * @param {Error} err
     * @param {Stats} result
     */
    invokeMonitor(err, result) {
        this.monitor && this.monitor(err, result);
    }
    /**
     * Handler
     * @param {Error} err
     * @param {Stats} result
     */
    handler(err, result) {
        this.invokeMonitor(err, result);
    }
}
exports.BuilderBase = BuilderBase;
