/// <reference types="node" />
import webpack, { Configuration, Stats } from 'webpack';
import { EventEmitter } from 'events';
import { GhostFileActions } from './ghost-file';
export type BuilderMonitor = (err?: Error, result?: Stats) => void;
type Mode = 'development' | 'production';
export type ConfigurationOptions = {
    mode: Mode;
    cwd: string;
    watchMode?: boolean;
};
/**
 * Wrapper for Webpack
 */
export declare class BuilderBase extends EventEmitter {
    compiler: webpack.Compiler;
    watching: any;
    monitor: BuilderMonitor;
    options: ConfigurationOptions;
    /**
     * Creates a new builder base instance
     * @param {EventEmitterOptions} options
     */
    constructor(options?: any);
    initCompiler(opts: ConfigurationOptions): void;
    /**
     * Start build process
     * @param {ConfigurationOptions} opts
     * @param {any=} ifs Input filesystem
     * @param {any=} ofs Output filesystem
     * @param {any=} wfs Watch filesystem
     */
    build(opts: ConfigurationOptions, ifs?: any, ofs?: any, wfs?: any): void;
    /**
     * Teardown logic
     * @return {Promise}
     */
    teardown(): Promise<unknown>;
    /**
     * Attaches a monitor that can be used to listen to events
     * @param {BuilderMonitor} mon
     */
    attachMonitor(mon: BuilderMonitor): void;
    /**
     * Gets input ghost files
     * @param {ConfigurationOptions} opts
     * @return {GhostFileActions[]}
     */
    getGhostFiles(opts: ConfigurationOptions): GhostFileActions[];
    /**
     * Supported events
     * @return {string[]}
     */
    eventNames(): string[];
    /**
     * Gets configuration
     * @param {ConfigurationOptions} opts
     * @return {Configuration}
     */
    getConfiguration(opts: ConfigurationOptions): Configuration;
    /**
     * Gets stylesheet test expression
     * @return {RegExp}
     */
    getStyleTestExp(): RegExp;
    /**
     * Gets LESS stylesheet test expression
     * @return {RegExp}
     */
    getLESSStyleTestExp(): RegExp;
    /**
     * Create alias mapping with peer dependencies
     * @param {string[]} dependencies
     * @param {string=} cwd Defaults to process.cwd()
     * @return {any}
     */
    mapPeerDependencies(dependencies: string[], cwd?: string): {
        [key: string]: string;
    };
    /**
     * Generate file from template / retreives optional file
     * @param {string} cwd Current Working Directory
     * @param {string} relativePath Relative path of the file from project root
     * @param {string} ejsFilePath Template file path
     * @param {object=} data (Optional) template render options
     * @return {string} Optional file from project dir / template output
     */
    getOptionalFile(cwd: string, relativePath: string, ejsFilePath: string, data?: any): string;
    /**
     * Invokes monitor if one's attached
     * @param {Error} err
     * @param {Stats} result
     */
    private invokeMonitor;
    /**
     * Handler
     * @param {Error} err
     * @param {Stats} result
     */
    private handler;
}
export {};
