"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendBuilder = void 0;
const webpack_node_externals_1 = __importDefault(require("webpack-node-externals"));
const builder_base_1 = require("./builder-base");
const path_1 = __importDefault(require("path"));
const webpack_virtual_modules_1 = __importDefault(require("webpack-virtual-modules"));
const generate_auto_loader_file_1 = require("./utils/generate-auto-loader-file");
const fs_1 = __importDefault(require("fs"));
/**
 * Backend Builder
 */
class BackendBuilder extends builder_base_1.BuilderBase {
    /**
     * Creates a new backend builder instance
     * @param {string} entryFilePath
     */
    constructor(entryFilePath) {
        super();
        this.entryFilePath = entryFilePath;
        this.virtualModules = new webpack_virtual_modules_1.default({
            [path_1.default.join('src', 'server.tsx')]: `
        // @ts-nocheck
        import React from 'react';
        import ReactDOM from 'react-dom';
        import { startApp } from '@magicjs.dev/frontend';
        import { initializeModules } from './auto-loader.tsx';
        import runApp from './app.tsx';

        initializeModules();
        runApp();
      `
        });
    }
    initCompiler(opts) {
        this.compiler.hooks.compilation.tap('MyPlugin', (compilation) => {
            const content = (0, generate_auto_loader_file_1.generateAutoloaderFile)(opts.cwd, 'backend');
            this.virtualModules.writeModule(path_1.default.join('src', 'auto-loader.tsx'), content);
        });
    }
    /**
     * @param {ConfigurationOptions} opts
     * @return {Configuration}
     */
    getConfiguration({ cwd, mode }) {
        var _a, _b, _c, _d;
        let packageJson = null;
        let serverExternalsAllowList = [];
        try {
            packageJson = JSON.parse(fs_1.default.readFileSync(path_1.default.join(cwd, 'package.json'), 'utf-8'));
            if (Array.isArray((_b = (_a = packageJson === null || packageJson === void 0 ? void 0 : packageJson.magicjsConfig) === null || _a === void 0 ? void 0 : _a.compilerOptions) === null || _b === void 0 ? void 0 : _b.serverBundleAllowList)) {
                serverExternalsAllowList = (_d = (_c = packageJson === null || packageJson === void 0 ? void 0 : packageJson.magicjsConfig) === null || _c === void 0 ? void 0 : _c.compilerOptions) === null || _d === void 0 ? void 0 : _d.serverBundleAllowList;
            }
        }
        catch (e) {
            console.error(e);
        }
        const babelLoaderOptions = {
            // Should not take any babelrc file located in the project root
            babelrc: false,
            inputSourceMap: mode === 'development',
            sourceMaps: mode === 'development' ? 'inline' : false,
            compact: mode === 'production',
            presets: [
                [
                    require.resolve('@babel/preset-env'),
                    {
                        targets: { browsers: ['last 2 versions'] },
                        modules: false,
                    },
                ],
                [
                    require.resolve('@babel/preset-typescript'),
                    { allowNamespaces: true },
                ],
                [require.resolve('@babel/preset-react')],
            ],
            cacheDirectory: true,
            cacheCompression: false,
            plugins: [
                require.resolve('@babel/plugin-proposal-class-properties'),
                require.resolve('@babel/plugin-syntax-dynamic-import'),
            ],
        };
        return {
            devtool: mode === 'development' ? 'source-map' : false,
            context: cwd,
            mode,
            resolve: {
                modules: ['scripts', 'node_modules'],
                extensions: ['.json', '.ts', '.tsx', '.js', '.jsx'],
                alias: Object.assign({}, this.mapPeerDependencies(['react', 'react-dom', 'react-router-dom'], cwd)),
                symlinks: true,
            },
            entry: {
                ['server']: this.entryFilePath
            },
            output: {
                publicPath: '/',
                filename: 'main.js',
                path: path_1.default.resolve(cwd, 'build', 'server'),
                assetModuleFilename: './assets/[hash][ext][query]',
            },
            target: 'node',
            externals: [
                (0, webpack_node_externals_1.default)({
                    allowlist: [
                        '@magicjs.dev/backend',
                        '@magicjs.dev/frontend',
                        ...serverExternalsAllowList,
                        // Allow the stylesheet to be handled by ignore-loader
                        this.getStyleTestExp(),
                    ],
                }),
            ],
            stats: {
                loggingTrace: false,
                errorStack: false,
            },
            plugins: [
                this.virtualModules
            ],
            module: {
                rules: [
                    {
                        test: /\.js$/,
                        enforce: 'pre',
                        use: [require.resolve('source-map-loader')],
                    },
                    {
                        test: /\.(js|mjs|jsx|ts|tsx)$/,
                        exclude: /node_modules/,
                        use: [
                            {
                                loader: require.resolve('babel-loader'),
                                options: babelLoaderOptions,
                            },
                        ],
                    },
                    {
                        test: /\.svg(\?v=\d+\.\d+\.\d+)?$/,
                        use: [
                            {
                                loader: require.resolve('babel-loader'),
                                options: babelLoaderOptions,
                            },
                            {
                                loader: require.resolve('@svgr/webpack'),
                                options: {
                                    babel: false,
                                    icon: true,
                                },
                            },
                        ],
                    },
                    {
                        test: /\.(png|jpg|jpeg|gif|woff2|woff|ttf|eot)$/i,
                        use: [
                            {
                                loader: require.resolve('file-loader'),
                                options: {
                                    name: '[contenthash].[ext]',
                                    outputPath: 'assets',
                                    emitFile: false,
                                },
                            },
                        ],
                    },
                    {
                        test: this.getStyleTestExp(),
                        loader: require.resolve('ignore-loader'),
                    },
                    {
                        test: this.getLESSStyleTestExp(),
                        loader: require.resolve('ignore-loader'),
                    },
                ],
            },
        };
    }
}
exports.BackendBuilder = BackendBuilder;
